package jsquid.xml;

import java.awt.Color;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.Set;
import java.util.TreeSet;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import jsquid.graph.JSquidGraph;

import medusa.graph.Graph;
import medusa.graph.UniqueEdge;

import org.w3c.dom.Document;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.*;




/***
 * 
 * @author martin and sanjit
 * 
 * Checks and parses the input XML file
 */
public class XMLDataLoader {

	//indecates whether the XML file is valid or not
	private boolean success = false;
	private Document doc = null;
	//the set of different types of edges
	Set edgetypes = new TreeSet();
	//list of different grouping types
	ArrayList nodeGroups = null;
	//holding the legend items describing the meaning of nodes' shape and color
	ArrayList legendItems= null;
	float confidenceCutoff = -1;
	float connectionCutoff = -1;
	//display settings varaibales
	boolean distributeOnInit = true;
	boolean showMedusaStyle = false;
	boolean showNodeLabels = true;
	boolean showEdgeLabels = false;
	boolean showNodeSize = true;
	boolean showUniformNodes = false;
	boolean showEdgeConf = true;
	boolean showGroupLabels = true;
	//there is one special edge type id which represents the summed up confidences between 
	//two distinct nodes. This id has the value 0 by default, if not defined in a different
	//way within the XML file
	private int summedConfidenceID = 0;
	//the nodes' X and Y positions are normalized between 0 and 1. This value multiplies the 
	//coordinates to spread them on the panel
	private static final int INITIAL_POS_MULTIPLIER = 600;
	//if one node does NOT belong to any group of a certain grouping type, it will be
	//assigned to this undefined group
	public static final String UNDEFINDED_GRP_CONST = "Undefined";
	//the subcellular location grouping is displayed in a speciel way with biologically shaped
	//grouping clusters. To be recognized as the location grouping type, it has to have this value
	//in the XML file
	public static final String LOCATION_GRP_CONST = "Sub-Cell Location";
	Graph g = null;
	
	public XMLDataLoader(String schemaPath, InputStream fileStream) throws XMLFileNotValidException{
		validateFile(schemaPath, fileStream);
	}
	
	/***
	 * 
	 * @param schemaPath
	 * @param fileStream
	 * @throws XMLFileNotValidException
	 * 
	 * checks, if the input XML file is valid to the XML Schema that comes with this program.
	 * if it is not valid, an Exception will be thrown and the program will not be executed any more
	 */
	private void validateFile(String schemaPath, InputStream fileStream) throws XMLFileNotValidException{
		DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();

		factory.setValidating(true);

		//setting the schema language
		factory.setAttribute(
				"http://java.sun.com/xml/jaxp/properties/schemaLanguage",
				"http://www.w3.org/2001/XMLSchema");
		//setting the schema file
		factory.setAttribute(
				"http://java.sun.com/xml/jaxp/properties/schemaSource",
				schemaPath);
		Exception exToThrow = null;
		try {
			DocumentBuilder parser = factory.newDocumentBuilder();
			parser.setErrorHandler(new XMLErrorHandler());
			doc = parser.parse(fileStream);
			success = true;
		} catch (ParserConfigurationException e) {
			System.out.println(e.getMessage());
			exToThrow = e;
		} catch (SAXException e) {
			System.out.println(e.getMessage());
			exToThrow = e;
		} catch (IOException e) {
			System.out.println(e.getMessage());
			exToThrow = e;
		}
		if(!success) {
			System.out.println("xml file INVALID");	
			throw new XMLFileNotValidException(exToThrow);
		}
	}
	
	/**
	 * 
	 * @return the graph object containing the data from the XML file
	 * 
	 * performs the operations needed to load the data from the XML file
	 */
	public Graph loadParameters() {
		if (success) {
			g = new JSquidGraph();
			summedConfidenceID = getSumID();
			loadSettings();
			loadConfidenceSettings();
			loadHyperlinkStubs();
			loadNodegroups();
			loadNodes();
			loadEdges();
		}
		
		return g;
	}
	
	/**
	 * 
	 *loads the hyperlink stubs from the XML file. One stub always consists of
	 *an URL prefix and might also consist of an URL postfix
	 */
	private void loadHyperlinkStubs(){
		HashMap hyperlinkMap = new HashMap();
		
		NodeList hyperlinkList = doc.getElementsByTagName("stub");
		for (int i = 0; i < hyperlinkList.getLength(); ++i) {
			Node hlNode = hyperlinkList.item(i);
			NamedNodeMap map = hlNode.getAttributes();
			
			//get the name attribute
			Node attHyperlinkName = map.getNamedItem("name");
			String name = attHyperlinkName.getNodeValue();
			
			//get the urlbegin attribute
			Node attHyperlinkURLBegin = map.getNamedItem("urlbegin");
			String urlbegin = attHyperlinkURLBegin.getNodeValue();
			
			//get the urlend attribute, if exists
			String urlend = null;
			try {
				Node attHyperlinkURLEnd = map.getNamedItem("urlend");
				urlend = attHyperlinkURLEnd.getNodeValue();
			} catch (Exception ex) {
				//optional attribute does not exist
			}
			
			LinkURL url = new LinkURL(name, urlbegin, urlend);
			hyperlinkMap.put(name, url);
		}
		g.setLinkStubMap(hyperlinkMap);
	}
	
	/**
	 * loads the grouping types from the XML file
	 *
	 */
	private void loadNodegroups(){
		nodeGroups = new ArrayList();
		NodeList nodegroupList = doc.getElementsByTagName("nodegroup");
		for (int i = 0; i < nodegroupList.getLength(); ++i) {
			Node nodegroupNode = nodegroupList.item(i);
			NamedNodeMap map = nodegroupNode.getAttributes();
			
			//get the name attribute
			Node attNodeName = map.getNamedItem("name");
			String name = attNodeName.getNodeValue();
			
			nodeGroups.add(name);
		}
		nodeGroups.add("Confidence");
	}
	
	/**
	 * loads the grouping types from the XML file
	 *
	 */
	private void loadLegend(){
		legendItems = new ArrayList();
		NodeList legendItemList = doc.getElementsByTagName("legendItem");
		for (int i = 0; i < legendItemList.getLength(); ++i) {
			Node item = legendItemList.item(i);
			NamedNodeMap map = item.getAttributes();
			
			//get the name attribute
			Node attNodeName = map.getNamedItem("name");
			String name = attNodeName.getNodeValue();
			
			//get the color attribute
			Node attNodeColor = map.getNamedItem("color");
			String[] colorRGB = attNodeColor.getNodeValue().split(",");
			int colorR = Integer.parseInt(colorRGB[0]);
			int colorG = Integer.parseInt(colorRGB[1]);
			int colorB = Integer.parseInt(colorRGB[2]);
		    Color color = new Color(colorR, colorG, colorB);
			
			//get the shape attribute
			Node attNodeShape = map.getNamedItem("shape");
			int shape = Integer.parseInt(attNodeShape.getNodeValue());
			
			legendItems.add(new LegendItem(name, color, shape));
		}
	}
	
	/**
	 * 
	 * @return the Array of grouping types
	 */
	public String[] getNodeGroupArray() {
		if (nodeGroups == null)
			loadNodegroups();
		return (String[])nodeGroups.toArray(new String[nodeGroups.size()]);
			
	}
	
	/**
	 * loads the nodes from the XML file
	 *
	 */
	private void loadNodes() {
		NodeList nodeList = doc.getElementsByTagName("node");
		for (int i = 0; i < nodeList.getLength(); ++i) {
			Node nodeNode = nodeList.item(i);
			NamedNodeMap map = nodeNode.getAttributes();
			
			//get the name attribute
			Node attNodeName = map.getNamedItem("name");
			String name = attNodeName.getNodeValue();
			
			//get the x coordinate attribute
			Node attNodeX = map.getNamedItem("x");
			double x = Double.parseDouble(attNodeX.getNodeValue()) * INITIAL_POS_MULTIPLIER;
			
			//get the y coordinate attribute
			Node attNodeY = map.getNamedItem("y");
			double y = Double.parseDouble(attNodeY.getNodeValue()) * INITIAL_POS_MULTIPLIER;
			
			//get the color of the node
			Node attNodeColor = map.getNamedItem("color");
			String[] colorRGB = attNodeColor.getNodeValue().split(",");
			int colorR = Integer.parseInt(colorRGB[0]);
			int colorG = Integer.parseInt(colorRGB[1]);
			int colorB = Integer.parseInt(colorRGB[2]);
			
			//get the shape id of the node
			Node attNodeShape = map.getNamedItem("shape");
			int shape = Integer.parseInt(attNodeShape.getNodeValue());
			
			medusa.graph.Node medusaNode = new medusa.graph.Node(name,x,y,shape,new Color(colorR, colorG, colorB));
			
			//get the size of the node, if exists
			try {
				Node attNodeSize = map.getNamedItem("size");
				int size = Integer.parseInt(attNodeSize.getNodeValue());
				medusaNode.setNodeSize(size);
			} catch (Exception ex) {
				//optional attribute does not exist
			}
			
			String annotation = null;
			HashMap linkIDmap = null;
			ArrayList synList = null;
			ArrayList groupList = null;
			
			//will finally hold the grouping types, this node is NOT part of
			ArrayList noInstanceNodeGroups = new ArrayList(nodeGroups); 
			
			//get al the child elements of one node element
			//children can be: Annotation, Groups, Hyperlink ids, Synonyms
			NodeList elements = nodeNode.getChildNodes();
			for (int j = 0; j < elements.getLength(); ++j) {
				Node element = elements.item(j);
				
				//get the annotation and repalace masked characters
				if (element.getNodeName().equals("att")){
					annotation = getTextContent(element);
					annotation = annotation.replaceAll("&lt;", "<");
					annotation = annotation.replaceAll("&gt;", ">");	
				}
				
				//get the groups this node belongs to 
				else if (element.getNodeName().equals("groups")) {
					groupList = new ArrayList();
					NodeList groups = element.getChildNodes();
					for (int k = 0; k < groups.getLength(); ++k) {
						Node group = groups.item(k);
						if (group.getNodeName().equals("group")) {
							NamedNodeMap gMap = group.getAttributes();
							
							//get the grouping type
							Node attNodeGroupRef = gMap.getNamedItem("ref");
							String groupRef = attNodeGroupRef.getNodeValue();
							
							//get the name of the group
							Node attNodeGroupName = gMap.getNamedItem("name");
							String groupName = attNodeGroupName.getNodeValue();
							
							groupList.add(groupName);
							
							//a grouping type is only valid, if it was declared within the
							//tag <nodegroups> - this is checked here 
							if (nodeGroups.contains(groupRef)) {
								//removing the grouping type, if this node takes part in a
								//group of this grouping type
								noInstanceNodeGroups.remove(groupRef);
								HashMap groupTable = g.getGroup(groupRef);
								if (groupTable == null)
									groupTable = new HashMap();
								
								ArrayList nodes;
								if (groupTable.containsKey(groupName))
									nodes = (ArrayList) groupTable.get(groupName);
								else
									nodes = new ArrayList();
					
								nodes.add(medusaNode);
								
								groupTable.put(groupName, nodes);
								g.setGroup(groupRef, groupTable);
							}
						}
					}
				}
				
				//get the hyperlink ids (the middle part of an LinkURL object)
				else if (element.getNodeName().equals("hlIDs")) {
					NodeList ids = element.getChildNodes();
					linkIDmap = new HashMap();
					for (int k = 0; k < ids.getLength(); ++k) {
						Node id = ids.item(k);
						if (id.getNodeName().equals("ID")) {
							NamedNodeMap iMap = id.getAttributes();
							
							//get the URL this id belongs to
							Node attNodeLinkIDRef = iMap.getNamedItem("ref");
							String linkIDRef = attNodeLinkIDRef.getNodeValue();
							
							//the actual link id (the URL's middle part)
							Node attNodeLinkIDName = iMap.getNamedItem("name");
							String linkIDName = attNodeLinkIDName.getNodeValue();
							
							ArrayList valueList = null;
							if (linkIDmap.containsKey(linkIDRef)) {
								valueList = (ArrayList)linkIDmap.get(linkIDRef);
								valueList.add(linkIDName);
							}
							else {
								valueList = new ArrayList();
								valueList.add(linkIDName);
								linkIDmap.put(linkIDRef, valueList);
							}
						}
					}
				}
				
				//get the synonymous names for this node
				else if (element.getNodeName().equals("syns")) {
					NodeList syns = element.getChildNodes();
					synList = new ArrayList();
					for (int k = 0; k < syns.getLength(); ++k) {
						Node syn = syns.item(k);
						if (syn.getNodeName().equals("syn")) {
							NamedNodeMap sMap = syn.getAttributes();
							
							//get the synonym
							Node attNodeSynName = sMap.getNamedItem("name");
							String synName = attNodeSynName.getNodeValue();
							
							synList.add(synName);
						}
					}
				}
				
			}
			
			//assignes the node to the undefined group for every grouping type in the list
			for (int k = 0; k < noInstanceNodeGroups.size(); ++k) {
				String groupRef = (String)noInstanceNodeGroups.get(k);
				HashMap groupTable = g.getGroup(groupRef);
				if (groupTable == null)
					groupTable = new HashMap();
				
				ArrayList nodes;
				if (groupTable.containsKey(UNDEFINDED_GRP_CONST))
					nodes = (ArrayList) groupTable.get(UNDEFINDED_GRP_CONST);
				else
					nodes = new ArrayList();
	
				nodes.add(medusaNode);
				
				groupTable.put(UNDEFINDED_GRP_CONST, nodes);
				g.setGroup(groupRef, groupTable);
			}
			
			medusaNode.setAnnotation(annotation);
			medusaNode.setLinkIDMap(linkIDmap);
			medusaNode.setSynonyms(synList);
			medusaNode.setGroups(groupList);
			g.addNode(medusaNode);
		}
			
	}
	
	/**
	 * @return the whole text content of a node
	 * @param a node
	 * recursive implementation to get the whole text content of a node(and its subnodes) 
	 */
	private String getTextContent(Node node) {
		Node child;
		String content = "";
		if (node.getNodeValue() != null)
			content += node.getNodeValue();
		
		NodeList nodes = node.getChildNodes();
		for(int i = 0; i < nodes.getLength(); i++) {
			child = nodes.item(i);
			if (child.getNodeValue() != null)
				content += child.getNodeValue();
			if(nodes.item(i).getChildNodes().getLength() > 0)
				content += getTextContent(nodes.item(i));
		}

		return content;
	}
	
	/***
	 * loads the edges from the xml file
	 *
	 */
	private void loadEdges() {
		NodeList edgeList = doc.getElementsByTagName("edge");
		for (int i = 0; i < edgeList.getLength(); ++i) {
			Node edgeNode = edgeList.item(i);
			NamedNodeMap map = edgeNode.getAttributes();
			
			//load the name of the first node this edge connects
			Node attNodeFirstNode = map.getNamedItem("n1");
			String firstNode = attNodeFirstNode.getNodeValue();
			
			//load the name of the second node this edge connects
			Node attNodeSecondNode = map.getNamedItem("n2");
			String secondNode = attNodeSecondNode.getNodeValue();
			
			//load the interaction type of this edge
			Node attNodeType = map.getNamedItem("iType");
			int type = Integer.parseInt(attNodeType.getNodeValue());
			
			//load the confidence value of this node
			Node attNodeConfidence = map.getNamedItem("conf");
			float confidence = Float.parseFloat(attNodeConfidence.getNodeValue());
			
			//load the oriantation (determines how much the edge will be curved)
			Node attNodeOrientation = map.getNamedItem("ortn");
			double orientation = Double.parseDouble(attNodeOrientation.getNodeValue());
			
			medusa.graph.Edge medusaEdge = null;
			if (type == summedConfidenceID) {
				medusaEdge = new medusa.graph.Edge(firstNode, secondNode, confidence, type, orientation);
				g.addSingleEdge(medusaEdge);
			}
			else {
				medusaEdge = new medusa.graph.Edge(firstNode, secondNode, confidence, type, orientation);
				g.addEdge(medusaEdge);
				edgetypes.add(new Integer(type));
			}
		}
		//if no normal edges were found only single edges we have to create the unique edge
		//container for the layout algorithms
		if(g.getUniqueEdges().size() == 0){
			ArrayList uniqueEdges = new ArrayList();
			for(Iterator iter = g.getSingleEdges().iterator(); iter.hasNext();){
				medusa.graph.Edge e = (medusa.graph.Edge)iter.next();
				UniqueEdge ue = new UniqueEdge(e);
				uniqueEdges.add(ue);
			}
			g.setUniqueEdges(uniqueEdges);
		}
	}
	
	public Set getEdgetypes() {
		return edgetypes;
	}
	
	/**
	 * 
	 * @return the id for the edge id representing the summed confidence
	 * 
	 * this method will get the edge id for the summed-up confidence
	 * if it not exists, it will return zero (default value)
	 */
	private int getSumID() {
		int id = 0;
		try {
			NodeList sumElementList = doc.getElementsByTagName("sum");
			if (sumElementList.getLength() == 1) {
				Node node = sumElementList.item(0);
				NamedNodeMap map = node.getAttributes();
				
				Node sumNodeID = map.getNamedItem("ID");
				id = Integer.parseInt(sumNodeID.getNodeValue());
			}
		} catch (Exception ex) {
			//optional element does not exist
		}
		return id;
	}
	
	/***
	 * 
	 * @param interactionClass (e.g. evidence by type, evidence by species or predicted links)
	 * @return the Hashtable containing the edge type id and the corresponding interaction name
	 *
	 *to show only interactions(edges types) of a distinct interaction class (e.g. evidence by species)
	 */
	public Hashtable getInteractionTable(String interactionClass) {
		Hashtable table = new Hashtable();
		try {
			NodeList interactionList = doc.getElementsByTagName(interactionClass);
			for (int i = 0; i < interactionList.getLength(); ++i) {
				Node edgeNode = interactionList.item(i);
				NamedNodeMap map = edgeNode.getAttributes();
				
				//get the id of the interaction type
				Node attNodeID = map.getNamedItem("ID");
				int id = Integer.parseInt(attNodeID.getNodeValue());
				
				//get the name of the interaction type
				Node attNodeName = map.getNamedItem("name");
				String name = attNodeName.getNodeValue();
				
				table.put(new Integer(id), name);
			}
		} catch (Exception ex) {
			//invalid interaction class
		}
		return table;
	}
	
	/**
	 * 
	 * @param interactionClass (e.g. evidence by type, evidence by species or predicted links)
	 * @return the Hashtable containing the edge type id and the corresponding color
	 *
	 *retruns a Hashtable which holds the specific color of each interaction type of this
	 *specific interaction class
	 */
	public Hashtable getColorTable(String interactionClass) {
		Hashtable table = new Hashtable();
		try {
			NodeList interactionList = doc.getElementsByTagName(interactionClass);
			for (int i = 0; i < interactionList.getLength(); ++i) {
				Node edgeNode = interactionList.item(i);
				NamedNodeMap map = edgeNode.getAttributes();
				
				//get the id of the interaction type
				Node attNodeID = map.getNamedItem("ID");
				int id = Integer.parseInt(attNodeID.getNodeValue());
				
				//get the color
				Node attNodeColor = map.getNamedItem("color");
				String[] colorRGB = attNodeColor.getNodeValue().split(",");
				int colorR = Integer.parseInt(colorRGB[0]);
				int colorG = Integer.parseInt(colorRGB[1]);
				int colorB = Integer.parseInt(colorRGB[2]);
				
				table.put(new Integer(id), new Color(colorR, colorG, colorB));
			}
		} catch (Exception ex) {
			//invalid interaction type
		}
		return table;
	}
	
	
	/**
	 * 
	 * @param interactionClass (e.g. evidence by type, evidence by species or predicted links)
	 * @return the Hashtable containing the edge type id and the information if the checkboxes
	 * corresponding to this interaction type are checked in the EdgeMultiPanel and further if edges of
	 * of this type should be displayed from the beginning 
	 *
	 *returns a Hashtable which holds the specific checked information for the
	 *specific interaction class
	 */
	public Hashtable getCheckedTable(String interactionClass) {
		Hashtable table = new Hashtable();
		NodeList interactionList = doc.getElementsByTagName(interactionClass);
		for (int i = 0; i < interactionList.getLength(); ++i) {
			Node node = interactionList.item(i);
			NamedNodeMap map = node.getAttributes();

			//get the interaction type id 
			Node attNodeID = map.getNamedItem("ID");
			int id = Integer.parseInt(attNodeID.getNodeValue());

			//get the checked flag - true by default
			boolean checked = true;
			try {
				Node attNodeChecked = map.getNamedItem("checked");
				checked = Boolean.valueOf(attNodeChecked.getNodeValue()).booleanValue();
			} catch (Exception ex) {
				// optional attribute does not exist
			}
			table.put(new Integer(id), new Boolean(checked));
		}
		
		return table;
	}
	
	/**
	 * loads the confidence settings from the xml file
	 * these settings are used for the confidence grouping
	 * if there is no such information in the xml file, default values are applied
	 *
	 */
	private void loadConfidenceSettings(){
		NodeList settingsList = doc.getElementsByTagName("confidenceSettings");
		for (int i = 0; i < settingsList.getLength(); ++i) {
			Node settingsNode = settingsList.item(i);
			NamedNodeMap map = settingsNode.getAttributes();
			
			//get the confidence cutoff
			Node confCutoffNode = map.getNamedItem("confidenceCutoff");
			float confCutoff = Float.parseFloat(confCutoffNode.getNodeValue());
			
			//get the connection cutoff
			Node connCutoffNode = map.getNamedItem("connectionCutoff");
			float connCutoff = Float.parseFloat(connCutoffNode.getNodeValue());
			
			this.confidenceCutoff = confCutoff;
			this.connectionCutoff = connCutoff;
		}	
	}
	
	public void loadSettings() {
		NodeList settingsList = doc.getElementsByTagName("settings");
		for (int i = 0; i < settingsList.getLength(); ++i) {
			Node settingsNode = settingsList.item(i);
			NamedNodeMap map = settingsNode.getAttributes();
			
			//get boolean value for distribution on init
			try {
				Node distOnInitNode = map.getNamedItem("distributeOnInit");
				boolean distOnInit = Boolean.parseBoolean(distOnInitNode.getNodeValue());
				this.distributeOnInit = distOnInit;
			} catch (Exception ex) {
				//optional attribute does not exist
			}
			
			//get boolean value for displaying graph in Meduasa style
			try {
				Node medusaStyleNode = map.getNamedItem("showMedusaStyle");
				boolean showMedusaStyle = Boolean.parseBoolean(medusaStyleNode.getNodeValue());
				this.showMedusaStyle = showMedusaStyle;
			} catch (Exception ex) {
				//optional attribute does not exist
			}
			
			//get boolean value for displaying node labels
			try {
				Node showNodeLabelsNode = map.getNamedItem("showNodeLabels");
				boolean showNodeLabels = Boolean.parseBoolean(showNodeLabelsNode.getNodeValue());
				this.showNodeLabels = showNodeLabels;
			} catch (Exception ex) {
				//optional attribute does not exist
			}
			
			//get boolean value for displaying edge labels
			try {
				Node showEdgeLabelsNode = map.getNamedItem("showEdgeLabels");
				boolean showEdgeLabels = Boolean.parseBoolean(showEdgeLabelsNode.getNodeValue());
				this.showEdgeLabels = showEdgeLabels;
			} catch (Exception ex) {
				//optional attribute does not exist
			}
			
			//get boolean value for displaying node size
			try {
				Node showNodeSizeNode = map.getNamedItem("showNodeSize");
				boolean showNodeSize = Boolean.parseBoolean(showNodeSizeNode.getNodeValue());
				this.showNodeSize = showNodeSize;
			} catch (Exception ex) {
				//optional attribute does not exist
			}
			
			//get boolean value for displaying uniform nodes
			try {
				Node showUniformNodeNode = map.getNamedItem("showUniformNodes");
				boolean showUniformNodes = Boolean.parseBoolean(showUniformNodeNode.getNodeValue());
				this.showUniformNodes = showUniformNodes;
			} catch (Exception ex) {
				//optional attribute does not exist
			}
			
			//get boolean value for displaying edge thickness according
			//to their confidence
			try {
				Node showEdgeConfNode = map.getNamedItem("showEdgeConf");
				boolean showEdgeConf = Boolean.parseBoolean(showEdgeConfNode.getNodeValue());
				this.showEdgeConf = showEdgeConf;
			} catch (Exception ex) {
				//optional attribute does not exist
			}
			
			//get boolean value for displaying group labels
			try {
				Node showGroupLabelsNode = map.getNamedItem("showGroupLabels");
				boolean showGroupLabes = Boolean.parseBoolean(showGroupLabelsNode.getNodeValue());
				this.showGroupLabels = showGroupLabes;
			} catch (Exception ex) {
				//optional attribute does not exist
			}
			
		}	
	}
	
	public ArrayList getLegend(){
		if (legendItems == null)
			loadLegend();
		
		return legendItems;
	}
	
	public float getConfidenceCutoff(){
		return this.confidenceCutoff;
	}
	
	public float getConnectionCutoff(){
		return this.connectionCutoff;
	}
	
	public boolean isDistributeOnInit() {
		return distributeOnInit;
	}
	
	public boolean isShowMedusaStyle() {
		return showMedusaStyle;
	}
	
	public boolean isShowNodeLabels() {
		return showNodeLabels;
	}
	
	public boolean isShowEdgeLabels() {
		return showEdgeLabels;
	}
	
	public boolean isShowNodeSize() {
		return showNodeSize;
	}
	
	public boolean isShowUniformNodes() {
		return showUniformNodes;
	}
	
	public boolean isShowEdgeConf() {
		return showEdgeConf;
	}
	
	public boolean isShowGroupLabels() {
		return showGroupLabels;
	}
}
