/**
 * Collapse.java
 */
package jsquid.graph;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;

import medusa.graph.Edge;
import medusa.graph.Node;
import medusa.graph.UniqueEdge;

/**
 * @author sanjit
 * Collapse and Expands nodes
 */
public class Collapse {

	private int collapseCounter = 0;

	//Map of collapsed Nodes key:name value:a CollapsedData object 
	private HashMap collapseMap = new HashMap(); 

	//EdgeMap of orginal edge List  
	private EdgeMap edgeMap = null;
	
	//EdgeMap of orginal single edge List
	private EdgeMap singleEdgeMap = null;

	private ArrayList edges;
	
	private ArrayList singleEdges;

	private HashMap nodes;
	
	private boolean isDetailedGraph;

	private ArrayList uniqueEdges;

	public Collapse(ArrayList singleEdges, ArrayList edges, HashMap nodes, ArrayList uniqueEdges, boolean isDetailedGraph) {
		this.edges = edges;
		this.singleEdges = singleEdges;
		this.nodes = nodes;
		this.uniqueEdges = uniqueEdges;
		this.isDetailedGraph = isDetailedGraph;
	}

	/**
	 * Add a edge @param e to the ArrayList @param edges
	 */
	public void addEdge(Edge e, ArrayList edges) {
		if ((!edges.contains(e)) & (e != null)) {
			edges.add(e);
			//add UniqueEdge if necessary
			UniqueEdge ue = new UniqueEdge(e);
			if (!uniqueEdges.contains(ue))
				uniqueEdges.add(ue);
		}
	}

	/**
	 * Remove edge @param e from ArrayList @param edges
	 */
	public void removeEdge(Edge e, ArrayList edges) {
		if (!edges.contains(e))
			return;
		edges.remove(e);
		//remove UniqueEdge if necessary
		UniqueEdge ue = new UniqueEdge(e);
		if (!edges.contains(ue))
			uniqueEdges.remove(ue);
	}

	/**
	 * @return all collapsed nodes in a ArrayList
	 */
	public ArrayList getCollapsed() {
		ArrayList collapsedNodes = new ArrayList();
		for (Iterator i = nodes.values().iterator(); i.hasNext();) {
			Node n = (Node) i.next();
			if (n.isCollapseNode())
				collapsedNodes.add(n);
		}
		return collapsedNodes;
	}

	// ------------------ Collapsing starts here------------------------
	/**
	 * Collapse all nodes which are selected
	 */
	public void collapseNodes() {

		//create orginal edgeMaps
		if (edgeMap == null) {
			edgeMap = new EdgeMap();
			edgeMap.createCompleteEdgeMatrix(edges);
		}
		if(singleEdgeMap == null){
			singleEdgeMap = new EdgeMap();
			singleEdgeMap.createCompleteEdgeMatrix(singleEdges);
		}

		CollapsedData data = new CollapsedData();
		String annotation = "";
		double x = 0;
		double y = 0;
		int edgeCount = 0;
		String name = "collapsed" + collapseCounter; //name of collapsed node
		//Iterate over all nodes
		for (Iterator iterNodes = nodes.keySet().iterator(); iterNodes
				.hasNext();) {
			String key = (String) iterNodes.next();
			Node n = (Node) nodes.get(key);
			if (n.isFixed()) {
				edgeCount += n.getEdgeCount();
				n.setFixed(false);
				n.setSearchHit(false);
				data.addDisplayNode(n.getLabel(), n);
				if (n.isCollapseNode()) {
					annotation += n.getAnnotation();
					//get all nodes which are collapsed in this node
					CollapsedData encapsulatedData = (CollapsedData) collapseMap
							.get(n.getLabel());
					data.getNodes().putAll(encapsulatedData.getNodes());
				} else {
					annotation += n.getLabel() + " ";
					data.addNode(n.getLabel(), n);
				}
				x += n.getX();
				y += n.getY();
				iterNodes.remove(); //remove selected nodes
			}
		}
		//create collapsed Node
		Node collapsed = new Node(name, x / data.getNodeSize(), y
				/ data.getNodeSize());
		collapsed.setEdgeCount(edgeCount);
		collapsed.setNodeSize(30);
		collapsed.setCollapseNode(true);
		collapsed.setDisplayLabel("( collapsed )");
		collapsed.setAnnotation(annotation);

		//create Edge List for extendes edges and singleEdges
		//if (isDetailedGraph)
			createNewEdgeList(data, collapsed, edges);
		//else
			createNewEdgeList(data, collapsed, singleEdges);
		//put CollapsedData into map
		collapseMap.put(name, data);
		nodes.put(name, collapsed);
		// sortEdgeList();
		++collapseCounter; //increment counter because every node needs a unique name
	}

	/**
	 * Create new Edge List 
	 * @param data CollapsedData
	 * @param collapsed new collapsed Node
	 * @param edges List of edges
	 */
	private void createNewEdgeList(CollapsedData data, Node collapsed, ArrayList edges) {
		ArrayList newEdges = new ArrayList();
		ArrayList oldEdges = new ArrayList();
		HashMap orientationEdgesMap = new HashMap(); //For calculation of orientations
		//Iterate over edges
		for (Iterator iterEdge = edges.iterator(); iterEdge.hasNext();) {
			Edge e = (Edge) iterEdge.next();
			int containsCounterNode1 = 0;
			int containsCounterNode2 = 0;
			//Check if edge is getting changed by new collapsed Node
			if (data.getDisplayNodes().containsKey(e.n1))
				++containsCounterNode1;
			if (data.getDisplayNodes().containsKey(e.n2))
				++containsCounterNode2;

			//Edge is obsolete because it is in the collapsed node 
			if (containsCounterNode1 == 1 && containsCounterNode2 == 1)
				oldEdges.add(e);
			//One Node of the edge is obsolete because it is in the collapsed node
			else if (containsCounterNode1 == 1) {
				oldEdges.add(e);
				Edge eNew = (Edge) e.clone();
				eNew.n1 = collapsed.getLabel();
				calculateOrientation(orientationEdgesMap, eNew);
				newEdges.add(eNew);
			} else if (containsCounterNode2 == 1) {
				oldEdges.add(e);
				Edge eNew = (Edge) e.clone();
				//switch node names so that n1 is collapsed node
				//easier for calculate Orientation
				String help = eNew.n1;
				eNew.n1 = collapsed.getLabel();
				eNew.n2 = help;
				calculateOrientation(orientationEdgesMap, eNew);
				newEdges.add(eNew);
			}
		}
		removeEdges(edges, oldEdges);
		addEdges(edges, newEdges);
	}

	/**
	 * calculate Orientation of edge
	 * @param orientationEdgesMap HashMap key: nodename, value: orientationList
	 * @param eNew edge to change
	 */
	private void calculateOrientation(HashMap orientationEdgesMap, Edge eNew) {
		ArrayList orientationList; // contains used orientations
		//get orientationList
		if (orientationEdgesMap.containsKey(eNew.n2))
			orientationList = (ArrayList) orientationEdgesMap.get(eNew.n2);
		else
			orientationList = new ArrayList();
		eNew.setOrientation(findOrientation(orientationList, eNew
				.getOrientation()));
		orientationList.add(new Double(eNew.getOrientation()));
		orientationEdgesMap.put(eNew.n2, orientationList);
	}

	/**
	 * find new Orientation which is not used in the orientation List
	 * @param orientationList List of orientation Values
	 * @param orientation current orientation
	 * @return new orientation
	 */
	private double findOrientation(ArrayList orientationList, double orientation) {
		double nextMin = orientation;
		double nextMax = orientation;

		while (orientationList.contains(new Double(orientation))) {
			//find new orientation 0.5 steps
			nextMin -= 0.5;
			nextMax += 0.5;
			if (!orientationList.contains(new Double(nextMin)))
				return nextMin;
			if (!orientationList.contains(new Double(nextMax)))
				return nextMax;
		}
		return orientation;
	}

	/**
	 * add @param edgesList to @param edges
	 */
	private void addEdges(ArrayList edges, ArrayList edgesList) {
		for (Iterator iterEdge = edgesList.iterator(); iterEdge.hasNext();) {
			Edge e = (Edge) iterEdge.next();
			addEdge(e, edges);
		}

	}

	/**
	 * remove @param edgesList from @param edges
	 */
	private void removeEdges(ArrayList edges, ArrayList edgesList) {
		for (Iterator iterEdge = edgesList.iterator(); iterEdge.hasNext();) {
			Edge e = (Edge) iterEdge.next();
			removeEdge(e, edges);
		}
	}

	// ------------------ Collapsing ends here------------------------
	// ------------------ Expanding start here------------------------

	/**
	 * Expand the Node @param toExpandNode
	 */
	public void expandNode(String toExpandNode) throws Exception {
		//get CollapsedData from toExpandNode
		CollapsedData data = (CollapsedData) collapseMap.get(toExpandNode);
		nodes.remove(toExpandNode);
		//restore nodes which where collapsed by toExpandNode
		nodes.putAll(data.getDisplayNodes());
		data.setVisible(false);
		//expand edgelist for edges and singleEdges
		//if (isDetailedGraph)
			expandEdgeList(toExpandNode, data, edges, edgeMap);
		//else
			expandEdgeList(toExpandNode, data, singleEdges, singleEdgeMap);
		// sortEdgeList();
	}

	/**
	 * expand EdgeList from @param toExpandNode
	 * @param data CollapsedData
	 * @param edges List of Edges
	 * @param edgeMap EdgeMap of orginal edges
	 * @throws Exception
	 */
	private void expandEdgeList(String toExpandNode, CollapsedData data, ArrayList edges, EdgeMap edgeMap)
			throws Exception {
		ArrayList toRemoveEdges = new ArrayList();
		for (Iterator iterEdge = edges.iterator(); iterEdge.hasNext();) {
			Edge e = (Edge) iterEdge.next();
			if (e.contains(toExpandNode))
				toRemoveEdges.add(e);
		}
		//remove old edges
		removeEdges(edges, toRemoveEdges);
		//find edges to create don't create edges if edgeMap is empty
		//than orginal graph has no edges
		if(edgeMap.getCompleteEdgeMatrix().size() != 0)
			findEdgesExpand(data, edges, edgeMap);

	}

	/**
	 * Find edges which should be created
	 * @param data CollapsedData
	 * @param edges List of Edges
	 * @param edgeMap EdgeMap of orginal edges
	 * @throws Exception
	 */
	private void findEdgesExpand(CollapsedData data, ArrayList edges, EdgeMap edgeMap) throws Exception {

		for (Iterator iterNodes = data.getDisplayNodes().values().iterator(); iterNodes
				.hasNext();) {
			String nameFrom = null;
			ArrayList nodesFrom = new ArrayList();
			Node n = (Node) iterNodes.next();
			//check if node is a collapsed Node
			if (n.isCollapseNode()) {
				nameFrom = n.getLabel();
				//get collapsedNode Data
				CollapsedData encapsulatedData = (CollapsedData) collapseMap
						.get(n.getLabel());
				nodesFrom = new ArrayList(encapsulatedData.getNodes().values());
			} else
				nodesFrom.add(n);
			addEdgesExpand(nameFrom, nodesFrom, edges, edgeMap);

		}
	}

	/**
	 * add edges to expand
	 * @param nameFrom name of encapsulated collapse node. is null if not a collapsednode
	 * @param nodesFrom node/s which should be added
	 * @param edges 
	 * @param edgeMap
	 * @throws Exception
	 */
	private void addEdgesExpand(String nameFrom, ArrayList nodesFrom, ArrayList edges, EdgeMap edgeMap)
			throws Exception {
		HashMap completeEdgeMatrix = edgeMap.getCompleteEdgeMatrix(); //Edge Matrix
		HashMap orientationEdgesMap = new HashMap();
		//Iterate over all newly added nodes
		for (Iterator iterNodes = nodesFrom.iterator(); iterNodes.hasNext();) {
			Node n = (Node) iterNodes.next();
			//get all edges of node
			ArrayList expandEdges = (ArrayList) completeEdgeMatrix.get(n
					.getLabel());
			//iterate over edges to expand
			if(expandEdges != null){
				for (Iterator iterEdges = expandEdges.iterator(); iterEdges
						.hasNext();) {
					Edge e = (Edge) iterEdges.next();
					Edge clone = (Edge) e.clone();
					Node complement = clone.getComplement(n);
					//check if complement is still available
					if (nodes.get(complement.getLabel()) == null) {
						String foundName = findComplement(complement.getLabel());
						if (foundName == null)
							throw new Exception(
									"Error in Expanding data not correct");
						clone.renameNode(complement.getLabel(), foundName);
					}
					//check if node is in a collapsedNode
					if (nameFrom != null) {
						clone.renameNode(n.getLabel(), nameFrom);
						if (!clone.n1.equals(nameFrom))
							clone.switchEdges();
						calculateOrientation(orientationEdgesMap, clone);
					}
					addEdge(clone, edges);
				}
			}
		}

	}

	/**
	 * find node (name = @param label) in collapsedNodes
	 * @param label
	 * @return Label of collapsedNode where node was found. null if data is corrupt
	 */
	private String findComplement(String label) {
		ArrayList collapsedNodes = getCollapsed();
		for (Iterator iterCollapsedNodes = collapsedNodes.iterator(); iterCollapsedNodes
				.hasNext();) {
			Node n = (Node) iterCollapsedNodes.next();
			CollapsedData data = (CollapsedData) collapseMap.get(n.getLabel());
			if (data.getNodes().get(label) != null)
				return n.getLabel();
		}
		return null; // Problem with the data throw Exception!
	}

}
