/**
 * XGMMLConverter.java
 */
package jsquid.export.xml;

import java.awt.Color;
import java.io.File;
import java.util.HashMap;
import java.util.Iterator;

import medusa.MedusaSettings;
import medusa.graph.Edge;
import medusa.graph.Graph;
import medusa.graph.Node;

import org.xml.sax.SAXException;
import org.xml.sax.helpers.AttributesImpl;

/**
 * @author sanjit XGMML Implementation of SAX Converter. Includes some Cytoscape
 *         only Elements.
 */
public class XGMMLConverter extends XMLSAXConverter {

	private HashMap idNodeMap; //Map with key: label and value: id

	public XGMMLConverter(File input, Graph graph, MedusaSettings settings, boolean prettyEdge) {
		super(input, graph, settings, prettyEdge);
	}

	protected void doConvert() throws Exception {
		AttributesImpl atts = new AttributesImpl();
		atts.addAttribute(null, null, "xmlns", null,
				XGMMLConstants.XMLNSNAMESPACE);
		atts.addAttribute(null, null, "label", null, "FunCoupNetwork");
		hd.startElement("", "", XGMMLConstants.STARTELEMENT, atts);

		createNodeElements();
		createEdgeElements();

		hd.endElement("", "", XGMMLConstants.STARTELEMENT);

	}

	/**
	 * Create Edge Elements
	 * 
	 * @throws SAXException
	 */
	private void createEdgeElements() throws SAXException {
		Iterator iterEdge = null;
		if(isPrettyEdge())
			iterEdge = getGraph().edgesIterator();
		else
			iterEdge = getGraph().getSingleEdges().iterator();
		for(;iterEdge.hasNext();) {
			Edge e = (Edge) iterEdge.next();
			Color color = getSettings().getColor(new Integer(e.getType()));
			if(e.getType() == 0)
				color = Color.DARK_GRAY;
			if (color != null) {

				Integer source = (Integer) idNodeMap.get(e.n1);
				Integer target = (Integer) idNodeMap.get(e.n2);

				AttributesImpl attEdge = new AttributesImpl();
				attEdge.addAttribute(null, null, "source", null, source
						.toString());
				attEdge.addAttribute(null, null, "target", null, target
						.toString());
				// confidence
				attEdge.addAttribute(null, null, "weight", null, Float
						.toString(e.getConf()));
				hd.startElement("", "", XGMMLConstants.EDGEELEMENT, attEdge);
				// Cytoscape Elements
				createAttElement("string", "interaction", getSettings()
						.getName(new Integer(e.getType())));
				createAttElement("string", "edge.toolTip", Float.toString(e
						.getConf()));
				createEdgeGraphics(color, e.getConf());

				hd.endElement("", "", XGMMLConstants.EDGEELEMENT);
			}

		}

	}

	/**
	 * Edge Graphics: color, width, fill, curved, ...
	 * 
	 * @param color
	 * @param confidence
	 * @throws SAXException
	 */
	private void createEdgeGraphics(Color color, float confidence)
			throws SAXException {
		AttributesImpl atts = new AttributesImpl();
		int width = 1;
		if (confidence >= 0.8)
			width = 6;
		else if (confidence >= 0.6)
			width = 4;
		else if (confidence >= 0.4)
			width = 2;

		atts.addAttribute(null, null, "fill", null, convertToHTMLColor(color));
		atts.addAttribute(null, null, "width", null, Integer.toString(width));
		hd.startElement("", "", XGMMLConstants.GRAPHICSELEMENT, atts);
		hd.startElement("", "", XGMMLConstants.ATT, null);
		// Cytoscape Elements
		createAttElement("", "curved", "", XGMMLConstants.CURVEDLINES);
		if (confidence < 0.2)
			createAttElement("", "edgeLineType", "",
					XGMMLConstants.EDGELINEDASH);
		hd.endElement("", "", XGMMLConstants.ATT);
		hd.endElement("", "", XGMMLConstants.GRAPHICSELEMENT);

	}

	/**
	 * Create Node Elements
	 * 
	 * @throws SAXException
	 */
	private void createNodeElements() throws SAXException {
		idNodeMap = new HashMap();
		int i = 1;
		for (Iterator iterNodes = getGraph().nodesIterator(); iterNodes
				.hasNext();) {
			Node n = (Node) iterNodes.next();
			AttributesImpl attNode = new AttributesImpl();
			idNodeMap.put(n.getLabel(), new Integer(i));

			attNode.addAttribute(null, null, "id", null, Integer.toString(i));
			attNode.addAttribute(null, null, "label", null, n.getLabel());
			hd.startElement("", "", XGMMLConstants.NODEELEMENT, attNode);
			String annotation = n.getAnnotation();
			if (annotation != null) {
				// Cytoscape Element
				createAttElement("string", "node.toolTip", annotation);
			}
			createNodeGraphics(n.getX(), n.getY(), n.getNodeSize(), n
					.getColor());
			hd.endElement("", "", XGMMLConstants.NODEELEMENT);
			++i;
		}

	}

	/**
	 * Create Node Graphics width, height, fill, ...
	 * 
	 * @param x
	 * @param y
	 * @param nodeSize
	 * @param color
	 * @throws SAXException
	 */
	private void createNodeGraphics(double x, double y, int nodeSize,
			Color color) throws SAXException {
		AttributesImpl atts = new AttributesImpl();
		nodeSize = Math.min(32 + nodeSize * 2, 50);
		atts.addAttribute(null, null, "w", null, Integer.toString(nodeSize));
		atts.addAttribute(null, null, "h", null, Integer.toString(nodeSize));
		atts.addAttribute(null, null, "x", null, Double.toString(x));
		atts.addAttribute(null, null, "y", null, Double.toString(y));
		atts.addAttribute(null, null, "fill", null, convertToHTMLColor(color));
		hd.startElement("", "", XGMMLConstants.GRAPHICSELEMENT, atts);
		hd.endElement("", "", XGMMLConstants.GRAPHICSELEMENT);
	}

	/**
	 * Converts a Color Obj to a HTML String
	 * 
	 * @param color
	 * @return
	 */
	private String convertToHTMLColor(Color color) {
		/*
		 * Integer[] colorHTML = new Integer[1]; colorHTML[0] = new
		 * Integer(color.getRGB()); String htmlColorString =
		 * String.format("%08X", colorHTML);
		 */
		String htmlColorString = Integer.toHexString(color.getRGB());
		htmlColorString = htmlColorString
				.substring(2, htmlColorString.length());
		htmlColorString = "#" + htmlColorString;
		return htmlColorString;
	}

	private void createAttElement(String type, String name, String value)
			throws SAXException {
		createAttElement(type, name, name, value);
	}

	/**
	 * Creates a Att Element with type, name, label and value If type, name,
	 * label or value == null. Attribute won't be displayed.
	 * 
	 * @param type
	 * @param name
	 * @param label
	 * @param value
	 * @throws SAXException
	 */
	private void createAttElement(String type, String name, String label,
			String value) throws SAXException {
		AttributesImpl atts = new AttributesImpl();
		if (type != null && type.length() != 0)
			atts.addAttribute(null, null, "type", null, type);
		if (name != null && name.length() != 0)
			atts.addAttribute(null, null, "name", null, name);
		if (label != null && label.length() != 0)
			atts.addAttribute(null, null, "label", null, label);
		if (value != null && value.length() != 0)
			atts.addAttribute(null, null, "value", null, value);
		hd.startElement("", "", XGMMLConstants.ATT, atts);
		hd.endElement("", "", XGMMLConstants.ATT);
	}

}
