/**
 * ExportToEPS.java
 */
package jsquid.export.picture;

import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.print.PageFormat;
import java.awt.print.Printable;
import java.awt.print.PrinterException;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.StringReader;
import java.io.UnsupportedEncodingException;
import java.util.Calendar;

import javax.print.Doc;
import javax.print.DocFlavor;
import javax.print.DocPrintJob;
import javax.print.PrintException;
import javax.print.SimpleDoc;
import javax.print.StreamPrintService;
import javax.print.StreamPrintServiceFactory;
import javax.print.DocFlavor.BYTE_ARRAY;
import javax.print.DocFlavor.SERVICE_FORMATTED;
import javax.print.attribute.HashPrintRequestAttributeSet;
import javax.print.attribute.PrintRequestAttributeSet;
import javax.print.attribute.standard.OrientationRequested;
import javax.print.event.PrintJobAdapter;
import javax.print.event.PrintJobEvent;

import medusa.display.BasicGraphPanel;



/**
 * @author sanjit
 * Implementation of PictureExporter for EPS Format
 * Creates PS data and add EPS header for EPS File.
 * The idea is basically from JaxoDraw(1.3-2) which is under
 * GNU General Public License as published by the Free Software Foundation;
 * either version 2 of the License, or (at our option) any later version.
 * 
 * At the option of JSquid GNU GPL Version 3 or (at our option) any later version 
 * of the License.
 * 
 * original authors:
 * @author Daniele Binosi <binosi@ect.it>
 * @author Lukas Theussl  <theussl@triumf.ca>
 */
public class ExportToEPS extends PictureExporter implements Printable {

	private double transX;

	private double transY;

	private double scale = 1.d;

	public ExportToEPS(File input, BasicGraphPanel panel) {
		super(input, panel);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see jsquid.export.PictureExporter#doExport()
	 */
	protected OutputStream doExport() throws IOException {
		return exportEPS();
	}

	/**
	 * print Method from Interface Printable
	 */
	public int print(Graphics graphics, PageFormat pageFormat, int pageIndex)
			throws PrinterException {

		transX = pageFormat.getImageableX();
		transY = pageFormat.getImageableY();

		double scaleX = pageFormat.getImageableWidth() / getPanel().getWidth();
		double scaleY = pageFormat.getImageableHeight()
				/ getPanel().getHeight();

		if (pageIndex == 0) {
			Graphics2D g2 = (Graphics2D) graphics;
			scale = scaleX;

			if (scaleY < scaleX) {
				scale = scaleY;
			}

			if (scale > 1.) {
				scale = 1.d;
			}

			g2.translate((double) transX, (double) transY);
			g2.scale(scale, scale);

			if (getPanel().getGraph() != null) {
				getPanel().paintNet(g2);
			}

			return Printable.PAGE_EXISTS;
		} else
			return Printable.NO_SUCH_PAGE;

	}

	/**
	 * Calculate BoundingBox for EPS File
	 * @return boundingbox
	 */
	public double[] getBB() {
		double minX = 50.;
		double minY = 250;
		if(getPanel().getHeight() < 400)
			minY += minY;
		else if(getPanel().getHeight() > 900)
			minY -= 100;
		
		double maxX = 560;
		if(getPanel().getWidth() < 400)
			maxX = 250;
		else if(getPanel().getWidth() < 550)
			maxX = 400;
		double maxY = 750;
		double[] bBox = { minX, minY, maxX, maxY };
		return bBox;
	}

	/**
	 * export Picture as EPS - Encapsulated PostScript
	 * @return
	 * @throws IOException
	 */
	public OutputStream exportEPS() throws IOException {
		
		String tempData = "";
		
		tempData = exportPS();

		BufferedReader inbr = null;
		BufferedWriter outbr = null;
		OutputStream stream = null;
		if (getFile() != null)
			stream = new FileOutputStream(getFile());

		try {
			inbr = new BufferedReader(new StringReader(tempData));
			outbr = new BufferedWriter(new OutputStreamWriter(stream));

			double[] bb = getBB();

			Calendar rightNow = Calendar.getInstance();

			String firstLine = "%!PS-Adobe-3.0 EPSF-3.0";
			String secondLine = "%%BoundingBox: " + (int) bb[0] + " "
					+ (int) bb[1] + " " + (int) bb[2] + " " + (int) bb[3];
			String thirdLine = "%%Creator: JSquid" + " with Version 1.0";
			String forthLine = "%%Title: functional coupling";
			String fifthLine = "%%CreationDate: "
					+ rightNow.get(Calendar.DAY_OF_MONTH) + "/"
					+ (rightNow.get(Calendar.MONTH) + 1) + "/"
					+ rightNow.get(Calendar.YEAR);

			String str;
			String setpagedevice = "setpagedevice";
			boolean isFirstLine = true;

			while ((str = inbr.readLine()) != null) {
				if (isFirstLine) {
					outbr.write(firstLine, 0, firstLine.length());
					outbr.newLine();
					outbr.write(secondLine, 0, secondLine.length());
					outbr.newLine();
					outbr.write(thirdLine, 0, thirdLine.length());
					outbr.newLine();
					outbr.write(forthLine, 0, forthLine.length());
					outbr.newLine();
					outbr.write(fifthLine, 0, fifthLine.length());
					outbr.newLine();
					isFirstLine = false;
				} else {
					/*
					 * check whether the line contains the postscript command
					 * "setpagedevice", which is not allowed in EPS
					 * 
					 * other forbidden commands: banddevice exitserver quit
					 * clear framedevice renderbands cleardictstack grestoreall
					 * setglobal copypage initclip setpagedevice erasepage
					 * initgraphics setshared initmatrix startjob
					 */
					if (str.indexOf(setpagedevice) != -1) {
						str = str.replaceAll(setpagedevice, "");
					}
					outbr.write(str, 0, str.length());
					outbr.newLine();
				}
			}
		} finally {
			if (inbr != null) {
				inbr.close();
			}
			if (outbr != null) {
				outbr.close();
			}
		}
		return stream;
	}

	/**
	 * export as PS - Postscript
	 * @return PS content
	 */
	private String exportPS() {
		DocFlavor flavor = SERVICE_FORMATTED.PRINTABLE;
		String psMimeType = BYTE_ARRAY.POSTSCRIPT.getMimeType();
		String tempData = "";
		StreamPrintServiceFactory[] factories = StreamPrintServiceFactory
				.lookupStreamPrintServiceFactories(flavor, psMimeType);
		if (factories.length == 0) {
			System.out.println("No Factory");
		} else {

			ByteArrayOutputStream bos = null;

			try {
				bos = new ByteArrayOutputStream();

				// Create a Stream printer for Postscript

				StreamPrintService sps = factories[0].getPrintService(bos);

				// Create and call a Print Job
				DocPrintJob job = sps.createPrintJob();

				PrintRequestAttributeSet aset = new HashPrintRequestAttributeSet();
				aset.add(OrientationRequested.PORTRAIT);

				Doc doc = new SimpleDoc(this, flavor, null);
				PrintJobWatcher pjDone = new PrintJobWatcher(job);
				job.print(doc, aset);
				pjDone.waitForDone();
				
				tempData = bos.toString("ISO-8859-1");

			} catch (PrintException pe) {
				System.out.println("Cannot print file");
			} catch (UnsupportedEncodingException e) {
				e.printStackTrace();
			} finally {
				try {
					if (bos != null) {
						bos.close();
					}
				} catch (IOException ioEx) {
					System.out.println("Cannot print file");
				}
			}
		}
		return tempData;
	}

	class PrintJobWatcher {
		// true if it is safe to close the print job's input stream
		private boolean done = false;

		PrintJobWatcher(DocPrintJob job) {
			// Add a listener to the print job
			job.addPrintJobListener(new PrintJobAdapter() {
				public final void printJobCanceled(PrintJobEvent pje) {
					allDone();
				}

				public final void printJobCompleted(PrintJobEvent pje) {
					allDone();
				}

				public final void printJobFailed(PrintJobEvent pje) {
					allDone();
				}

				public final void printJobNoMoreEvents(PrintJobEvent pje) {
					allDone();
				}

				void allDone() {
					synchronized (PrintJobWatcher.this) {
						done = true;
						PrintJobWatcher.this.notifyAll();
					}
				}
			});
		}

		public final synchronized void waitForDone() {
			try {
				while (!done) {
					super.wait();
				}
			} catch (InterruptedException e) {
				System.out.println("Print Job cancled");
			}
		}
	}

}
