package jsquid.display;

import java.awt.AlphaComposite;
import java.awt.Color;
import java.awt.Composite;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.Shape;
import java.awt.event.MouseEvent;
import java.awt.event.MouseMotionListener;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;

import javax.swing.BorderFactory;
import javax.swing.JPanel;

import jsquid.graph.JSquidGraph;


import medusa.display.BasicGraphPanel;
import medusa.display.PaintTools;
import medusa.graph.Edge;
import medusa.graph.Graph;
import medusa.graph.Node;

/**
 * @author martin
 * 
 * shows the small version of a graph in a panel. The realization of this class turned out to be quite tricky, since it
 * shares the same graph object with the JSquid panel and several corrective values are needed to make it work
 * properly. There have been several different approaches to solve this, but it still does not really work how it is
 * supposed to be. E.g. the overview rectangle (semi-transparent rectangle which indicates the current position on the JSquid panel)
 * should move over the graph rather than the graph over the rectangle.
 */
public class OverviewPanel extends JPanel implements MouseMotionListener {

	private Graph graph = null;

	private Graphics2D graphics = null;

	//scaling factor as in the JSquid panel
	private float scale = 1.0f;
	
	//correction value for zooming
	private float zoomOutScale = 1.0f;

	//the start point of the semi-transparent rectangle which indicates the viewable
	//area of the JSquid panel in the overview panel
	private Point overviewRect = null;

	//indicates how much smaller the overview graph appears compared with the original one
	private static final int OVERVIEW_SCALE = 10;
	
	//used for mous dragging event on the overview panel
	private Point posAtPrevCall = null;

	private BasicGraphPanel drawingPanelRef = null;

	public OverviewPanel() {
		this.setBackground(Color.white);
		this.setBorder(BorderFactory.createEtchedBorder());
		addMouseMotionListener(this);
	}

	public void setGraph(Graph g) {
		//this.graph = g.deepClone();
		this.graph = g;
	}

	public void setDrawingPanelRef(BasicGraphPanel panel) {
		this.drawingPanelRef = panel;
	}

	/***
	 * overrides JComponent's draw method
	 */
	public synchronized void paintComponent(Graphics g) {
		super.paintComponent(g);
		this.graphics = (Graphics2D) g;

		//init overview rectangle
		if (overviewRect == null) {
			overviewRect = new Point();
			overviewRect.x = getWidth() / 2
					- (drawingPanelRef.getWidth() / OVERVIEW_SCALE / 2);

			overviewRect.y = getHeight() / 2
					- (drawingPanelRef.getHeight() / OVERVIEW_SCALE / 2);
		}

		Iterator itor;
		//setting the correct iterator - simple or extended graph version
		if(drawingPanelRef.prettyEdge)
			itor = graph.edgesIterator();
		else 
			itor = graph.getSingleEdges().iterator();
		
		for (; itor.hasNext();) {
			Edge e = (Edge) itor.next();
			paintEdge(e);
		}

		for (itor = graph.nodesIterator(); itor.hasNext();) {
			Node node = (Node) itor.next();
			paintNode(node);
		}

		drawOverviewRect();
	}

	/**
	 * 
	 * @param scale
	 * 
	 * resizes the overview rectangle, if the original graph is scaled
	 * after a certain step it turned out to be better to rather shrink the graph in 
	 * the overview panel than to further increase the size of the overview rectangle
	 */
	public void resizeOverviewRect(double scale) {
		this.scale = (float)scale;
		if (scale < 1.0)
			this.zoomOutScale = (float)scale;
		else
			this.zoomOutScale = 1.0f;
	}

	/**
	 * draws the overvie rectangle according to several correction values
	 *
	 */
	private void drawOverviewRect() {
		Composite original = graphics.getComposite();
		graphics.setColor(Color.blue);

		Rectangle rect = new Rectangle(overviewRect.x, overviewRect.y,
				Math.round((drawingPanelRef.getWidth() / OVERVIEW_SCALE / scale * zoomOutScale)),
				Math.round((drawingPanelRef.getHeight() / OVERVIEW_SCALE / scale * zoomOutScale)));
		graphics.draw(rect);

		AlphaComposite ac = AlphaComposite.getInstance(AlphaComposite.SRC_OVER,
				0.1f);
		graphics.setComposite(ac);
		graphics.fill(rect);

		graphics.setComposite(original);
	}

	/**
	 * 
	 * @param e
	 * 
	 * paints one edge at the time - taking several correction values into account
	 */
	private void paintEdge(Edge e) {
		Node from = graph.getNode(e.getFromName());
		Node to = graph.getNode(e.n2);
		int x1 = Math.round((float)(from.getX() / OVERVIEW_SCALE / scale * zoomOutScale + overviewRect.x));
		int y1 = Math.round(((float)from.getY() / OVERVIEW_SCALE / scale * zoomOutScale + overviewRect.y));

		int x2 = Math.round(((float)to.getX() / OVERVIEW_SCALE / scale * zoomOutScale + overviewRect.x));
		int y2 = Math.round(((float)to.getY() / OVERVIEW_SCALE / scale * zoomOutScale + overviewRect.y));

		graphics.setPaint(Color.gray);
		PaintTools.paintPath(graphics, x1, y1, x2, y2, 0.0,
				BasicGraphPanel.offset, false);
	}

	/**
	 * 
	 * @param n
	 * 
	 *  paints one node at the time - taking several correction values into account
	 */
	private void paintNode(Node n) {
		Color c = n.getColor();
		int nodeSize = 6;

		int x = Math.round(((float)n.getX() / OVERVIEW_SCALE / scale * zoomOutScale + overviewRect.x))
				- nodeSize / 2;
		int y = Math.round(((float)n.getY() / OVERVIEW_SCALE / scale * zoomOutScale + overviewRect.y))
				- nodeSize / 2;
		int shapeID = n.getShape();

		Shape shape = PaintTools.getShape(shapeID, x, y, nodeSize);

		graphics.setPaint(c);
		graphics.fill(shape);

	}
	
	/**
	 * 
	 * @param xMove
	 * @param yMove
	 * 
	 * moves the graph in the original panel on the overview panel's mouse dragged event
	 */
	public void moveNodes(double xMove, double yMove) {
		Collection nodeTable = graph.getNodes().values();
		for (Iterator itor = nodeTable.iterator(); itor.hasNext();) {
			Node n = (Node) itor.next();
			if (n != null)
				n.move(xMove, yMove);
		}
		//move biologically shaped subcell. location group borders that do not contain any nodes.
		//we decided to also show empty subcell. location groups to make the cell look more recognizable
		ArrayList emptyOrganelleGroupNodes = ((JSquidGraph)graph).getEmptyOrganelleGroupNodes();
		if (emptyOrganelleGroupNodes != null) {
			for (Iterator itor = emptyOrganelleGroupNodes.iterator(); itor.hasNext();) {
				Node n = (Node) itor.next();
				if (n != null)
					n.move(xMove, yMove);
			}
		}
		repaint();
	}

	public void mouseDragged(MouseEvent e) {
		if (posAtPrevCall == null)
			posAtPrevCall = e.getPoint();

		Point actualPos = e.getPoint();
		double xDiff = (actualPos.getX() - posAtPrevCall.getX()) * OVERVIEW_SCALE * scale;
		double yDiff = (actualPos.getY() - posAtPrevCall.getY()) * OVERVIEW_SCALE * scale;

		moveNodes(xDiff, yDiff);

		drawingPanelRef.repaint();

		posAtPrevCall = actualPos;
		
	}

	public void mouseMoved(MouseEvent e) {
		//no need
	}

}