package jsquid.display;


import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;

import jsquid.graph.JSquidGraph;
import jsquid.xml.XMLDataLoader;

import medusa.graph.Graph;
import medusa.graph.Node;

/***
 * 
 * @author martin
 *
 *this class extends GroupDisplay and is specially designed for subcellular location grouping
 */
public class LocationGroupDisplay extends GroupDisplay {

	private int height;
	private double centerGrpSpace = RADIUSCONST;
	private double biggestGrpSpace = RADIUSCONST;
	
	private ArrayList emptyOrganelleNodeList = null;
	//to be able to also display organelle clusters which do not contain any nodes
	//it is necessary to introduce a dummy node for each of these clusters, otherwise it
	//would not be possible to mive and scale those properly
	private HashMap emptyOrganelleMap = new HashMap();
	
	public LocationGroupDisplay(int width, int height, Graph g) {
		super(width, g);
		this.height = height;
	}
	
	/**
	 * group Nodes according to the grouping type "Location"
	 */
	public void groupNodes(String type) {
		if (!type.equalsIgnoreCase(XMLDataLoader.LOCATION_GRP_CONST)){
			super.groupNodes(type);
			return;
		}
		HashMap groupTable = g.getGroup(type);
		
		emptyOrganelleNodeList = ((JSquidGraph)g).getEmptyOrganelleGroupNodes();
		if (emptyOrganelleNodeList == null)
			emptyOrganelleNodeList = new ArrayList();

		radius = RADIUSCONST;
		
		int centerPosX = width/2;
		int centerPosY = height/2;
		
		//group nodes that belong to the cytoplasm group
		groupOrganelle(groupTable, OrganelleType.CYTOPLASM_NAMES, centerPosX, centerPosY, OrganelleType.ORGANELLE_CYTOPLASM);
		
		int distToCenter = (int) Math.sqrt((centerGrpSpace*centerGrpSpace/2));
		
		//group nodes that belong to the endoplasmic reticulum group
		groupOrganelle(groupTable, OrganelleType.ER_NAMES, centerPosX-distToCenter, centerPosY-distToCenter, OrganelleType.ORGANELLE_ER);
		//group nodes that belong to the mitochondrion group
		groupOrganelle(groupTable, OrganelleType.MITO_NAMES, centerPosX+distToCenter, centerPosY-distToCenter, OrganelleType.ORGANELLE_MITO);
		//group nodes that belong to the golgi apparatus group
		groupOrganelle(groupTable, OrganelleType.GOLGI_NAMES, centerPosX+distToCenter, centerPosY+distToCenter, OrganelleType.ORGANELLE_GOLGI);
		//group nodes that belong to the nucleus group
		groupOrganelle(groupTable, OrganelleType.NUCLEUS_NAMES, centerPosX - distToCenter, centerPosY+distToCenter, OrganelleType.ORGANELLE_NUCLEUS);
		//group nodes that belong to the membrane group
		groupOrganelle(groupTable, OrganelleType.MEMBRANE_NAMES, centerPosX, centerPosY, OrganelleType.ORGANELLE_MEMBRANE);
		//group nodes that belong to the extracellular group
		groupOrganelle(groupTable, OrganelleType.EXTRACELL_NAMES, centerPosX-distToCenter, centerPosY+distToCenter, OrganelleType.ORGANELLE_EXTRACELL);
		//group nodes that belong to the undfined group
		groupOrganelle(groupTable, OrganelleType.UNDEF_NAMES, centerPosX+distToCenter, centerPosY+distToCenter, OrganelleType.ORGANELLE_UNDEF);
	
		((JSquidGraph)g).setEmptyOrganelleGroupNodes(emptyOrganelleNodeList);
	}
	
	/**
	 * 
	 * @param list of nodes
	 * @return the first occurring node and the number of nodes in the list
	 */
	private Object[] getFirstNodeAndNodecount(ArrayList list) {
		int nodeCount = 0;
		int j = 0;
		Node firstVisibleNode = null;
		for (Iterator itnodes = list.iterator(); itnodes.hasNext();) {
			Node n = (Node)itnodes.next();
			if (n.getEdgeCount() > 0) {
				++nodeCount;
				if (firstVisibleNode == null)
					firstVisibleNode = n;
			}
			++j;
		}
		Object[] result = {firstVisibleNode, new Integer(nodeCount)};
		return result;
	}
	
	/**
	 * 
	 * @param groupTable
	 * @param names
	 * @param startX
	 * @param startY
	 * @param organelle
	 * 
	 * group Nodes according to a distinct subcellular group
	 */
	private void groupOrganelle(HashMap groupTable, String[] names, double startX, double startY, String organelle) {
		radius = RADIUSCONST;
		boolean hasEmptyOrganelles = false;
		ArrayList nodes = new ArrayList();
		String name = "";
		for (int i = 0; i < names.length; ++i) {
			if (groupTable.containsKey(names[i])) {
				nodes = (ArrayList) groupTable.get(names[i]);
				name = names[i];
			}
		}
		
		Object[] firstNodeAndNodecount = getFirstNodeAndNodecount(nodes);
		Node firstVisibleNode = (Node)firstNodeAndNodecount[0];
		int nodeCount = ((Integer)firstNodeAndNodecount[1]).intValue();
		
		double groupSpace = 0.0;
		//ArrayList emptyOrganelleNodeList = ((JSquidGraph)g).getEmptyOrganelleGroupNodes();
		if (firstVisibleNode != null) {
			int groupSize = nodeCount - 1;

			double circleLayers = calculateCircleLayers(groupSize);
			//the space for the membrane is computed differently
			//it has to sircle arount all organelle shapes in the cell
			if (organelle.equals(OrganelleType.ORGANELLE_MEMBRANE))
				groupSpace = centerGrpSpace+(biggestGrpSpace*3);
			else
				groupSpace = calculateCircleSpace(Math.max(circleLayers, 1));
			//the cytoplasmic group is in the center
			if (organelle.equals(OrganelleType.ORGANELLE_CYTOPLASM))
				centerGrpSpace = Math.max(groupSpace, RADIUSCONST);
			//finding the biggest grouping space within the cell
			else if (!organelle.equals(OrganelleType.ORGANELLE_MEMBRANE) && 
					!organelle.equals(OrganelleType.ORGANELLE_EXTRACELL) &&
					!organelle.equals(OrganelleType.ORGANELLE_UNDEF)){
				if (biggestGrpSpace < groupSpace)
					biggestGrpSpace = Math.max(groupSpace, RADIUSCONST);
			}
		}
		else if (! organelle.equals(OrganelleType.ORGANELLE_EXTRACELL) && ! organelle.equals(OrganelleType.ORGANELLE_UNDEF)){
			hasEmptyOrganelles = true;
			double circleLayers = calculateCircleLayers(1);
			//the space for the membrane is computed differently
			//it has to sircle arount all organelle shapes in the cell
			if (organelle.equals(OrganelleType.ORGANELLE_MEMBRANE))
				groupSpace = centerGrpSpace+(biggestGrpSpace*3);
			else
				groupSpace = calculateCircleSpace(circleLayers);
			//the cytoplasmic group is in the center
			if (organelle.equals(OrganelleType.ORGANELLE_CYTOPLASM))
				centerGrpSpace = Math.max(groupSpace, RADIUSCONST);
			//finding the biggest grouping space within the cell
			else if (!organelle.equals(OrganelleType.ORGANELLE_MEMBRANE) && 
					!organelle.equals(OrganelleType.ORGANELLE_EXTRACELL) &&
					!organelle.equals(OrganelleType.ORGANELLE_UNDEF)){
				if (biggestGrpSpace < groupSpace)
					biggestGrpSpace = Math.max(groupSpace, RADIUSCONST);
			}
			firstVisibleNode = new Node(organelle);
			if (emptyOrganelleNodeList.contains(firstVisibleNode)) {
				int findPos = emptyOrganelleNodeList.indexOf(firstVisibleNode);
				firstVisibleNode = (Node)emptyOrganelleNodeList.get(findPos);
				
			}
			else {
				//firstVisibleNode = new Node(organelle);
				emptyOrganelleNodeList.add(firstVisibleNode);
				
				
			}
			ArrayList organelleNodes = new ArrayList(1);
			organelleNodes.add(firstVisibleNode);
			emptyOrganelleMap.put(organelle, organelleNodes);
			
			//((JSquidGraph)g).setEmptyOrganelleGroupNodes(emptyOrganelleNodeList);
			
			nodeCount = 1;
			name = organelle;
			firstVisibleNode.increaseEdgeCount();
		}
			
		GroupVariables groupVariables = new GroupVariables();

		groupVariables.setFirstNode(firstVisibleNode);
		groupVariables.setRadius(groupSpace);
		groupVariables.setGroupName(name);
		groupVariables.setNumberOfNodes(nodeCount);
		groupCollection.add(groupVariables);

		recalculateNodes(nodes, organelle, startX, startY, groupSpace);
		if (hasEmptyOrganelles)
			recalculateNodes((ArrayList)emptyOrganelleMap.get(organelle), organelle, startX, startY, groupSpace);
	}
	
	/**
	 * 
	 * @param nodes
	 * @param organelle
	 * @param startX
	 * @param startY
	 * @param groupSpace
	 * 
	 * recalculates the nodeposition for each node in a distinct group according to the start coordinate 
	 * and the list of nodes
	 */
	private void recalculateNodes(ArrayList nodes, String organelle, double startX, double startY, double groupSpace) {
		if (organelle.equals(OrganelleType.ORGANELLE_CYTOPLASM))
			calculateNewNodePositions(nodes, startX, startY);
		else if (organelle.equals(OrganelleType.ORGANELLE_ER))
			calculateNewNodePositions(nodes, startX - groupSpace, startY- groupSpace);
		else if (organelle.equals(OrganelleType.ORGANELLE_MITO))
			calculateNewNodePositions(nodes, startX + groupSpace, startY- groupSpace);
		else if (organelle.equals(OrganelleType.ORGANELLE_GOLGI))
			calculateNewNodePositions(nodes, startX + groupSpace, startY+ groupSpace);
		else if (organelle.equals(OrganelleType.ORGANELLE_NUCLEUS))
			calculateNewNodePositions(nodes, startX - groupSpace, startY+ groupSpace);
		else if (organelle.equals(OrganelleType.ORGANELLE_MEMBRANE))
			calculateNodePosOnMembrane(nodes, startX, startY, (int) groupSpace);
		else if (organelle.equals(OrganelleType.ORGANELLE_EXTRACELL))
			calculateNewNodePositions(nodes, startX - biggestGrpSpace * 2.5- groupSpace, startY + biggestGrpSpace * 2.5 + groupSpace);
		else
			// ORGANELLE_UNDEF
			calculateNewNodePositions(nodes, startX + biggestGrpSpace * 3- groupSpace, startY + biggestGrpSpace * 3 + groupSpace);
	}
	
	/**
	 * calculates the grouping space depending on the layers
	 */
	protected double calculateCircleSpace(double layers){
		return radius * (layers + 1) - shapeSize;
	}
	
	/**
	 * 
	 * @param nodes
	 * @param startX
	 * @param startY
	 * @param r
	 * 
	 * places the nodes belonging to the membrane group equally distributed on the membrane shape
	 */
	private void calculateNodePosOnMembrane(ArrayList nodes, double startX, double startY, int r) {
		int i = 0;
		double nodeCount = nodes.size();
		double tempStartX = startX;
		double tempStartY = startY;
		double xyArray[][] = calculatePointsOnCircle(r, startX,
				startY, nodeCount/6);
		for (Iterator itnodes = nodes.iterator(); itnodes.hasNext();) {
			Node n = (Node)itnodes.next();
			//set only if node is visible
			if (n.getEdgeCount() > 0) {
				tempStartX = xyArray[i][0];
				tempStartY = xyArray[i][1];
				n.setXY(tempStartX, tempStartY);
				++i;
			}
		}
	}
	
	/**
	 * Calculates points on a circle with @param radius and center @param startX and @param startY
	 * @return double array of points on circle
	 */
	protected double[][] calculatePointsOnCircle(int radius, double startX,
			double startY, double step) {
		double s = radius / step;
		double stepSize = s / radius;
		int arraySize = (int) ((2 * Math.PI) / stepSize);
		double xyArray[][] = new double[arraySize][2];
		int i = 0;
		for (double t = 0; t < (2 * Math.PI - stepSize); t = t + stepSize) {
			double Xcirc = radius * Math.cos(t);
			double Ycirc = radius * Math.sin(t);
			if(i < arraySize){
				xyArray[i][0] = Math.round(startX + Xcirc);
				xyArray[i][1] = Math.round(startY + Ycirc);
			}
			++i;
		}
		return xyArray;
	}
	
	/**
	 * Gets the name of a group for a certain x,y pair(for tooltip)
	 * @param x
	 * @param y
	 * @return name of group
	 */
	public String getGroup(int x, int y) {
		String toolTip = null;
		for (Iterator iter = groupCollection.iterator(); iter.hasNext();) {
			GroupVariables groupVariables = (GroupVariables) iter.next();
			Node node = groupVariables.getFirstNode();
			double r = groupVariables.getRadius();
			double dist = (node.getX() - x) * (node.getX() - x)
					+ (node.getY() - y) * (node.getY() - y);
			String groupName = groupVariables.getGroupName();
			//do not show tooltip for membrane (would be confusing)
			if (groupName.equalsIgnoreCase(OrganelleType.ORGANELLE_MEMBRANE))
				return null;
			if ((dist < (r * r))) {
				toolTip = "Group: " + groupName;
				return toolTip;
			}
		}
		return toolTip;
	}
}
