/**
 * GroupDisplay.java
 *
 */
package jsquid.display;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;

import jsquid.algorithm.ConfidenceSwitch;

import medusa.graph.Graph;
import medusa.graph.Node;

/**
 * @author sanjit and martin
 * This class displays the groups on the Display Panel(JSquidPanel).
 * Takes the groups from the graph and change the node position so that the fit the group
 * circle. 
 */
public class GroupDisplay {
	protected int width;

	protected static final int SHAPESIZECONST = 20;	//size of a avg. shape

	protected static final int RADIUSCONST = 50;	//minimum radius

	protected int layerSize = 8;	//how many nodes per circle

	protected int shapeSize = SHAPESIZECONST;

	protected int radius = RADIUSCONST;

	protected ArrayList groupCollection;	//Container of Meta/Information of grouping

	protected Graph g;

	public GroupDisplay(int width, Graph g) {
		this.width = width;
		groupCollection = new ArrayList();
		this.g = g;
	}

	/**
	 * group Nodes according to @param type
	 */
	public void groupNodes(String type) {
		//gets the hashmap from graph for type
		HashMap groupTable = g.getGroup(type);
		double borderCorrectionX = 0;	//circle center correction X
		double borderCorrectionY = 0;	//circle center correction Y
		double maxborderCorrection = 0;	//circe center correction depending on biggest circle
		double startX = 0;	//x coordinate of circle center
		double startY = 0;	//y coordinate of circle center

		//Iterate over all subgroups of this type
		for (Iterator itgroup = groupTable.keySet().iterator(); itgroup
				.hasNext();) {

			radius = RADIUSCONST;

			String groupName = itgroup.next().toString();
			ArrayList nodes = (ArrayList) groupTable.get(groupName); //Get nodes of subgroup

			int nodeCount = 0;
			int j = 0;
			Node firstVisibleNode = null;
			//find first Visible Node (perhaps some node are not visible due to confidence slider etc.).
			for (Iterator itnodes = nodes.iterator(); itnodes.hasNext();) {
				Node n = (Node)itnodes.next();
				if (n.getEdgeCount() > 0) {
					++nodeCount;
					if (firstVisibleNode == null)
						firstVisibleNode = n;
				}
				++j;
			}
			//check if there is any node in the subgroup which is visible
			if (firstVisibleNode != null) {
				int groupSize = nodeCount - 1;

				double circleLayers = calculateCircleLayers(groupSize);
				/*
				 * double circleSpace = (shapeSize + radius)
				 * (newborderCorrectionX + 1);
				 */
				double circleSpace = calculateCircleSpace(circleLayers);
				startX += borderCorrectionX + circleSpace;
				//if circle is bigger then width of panel draw next circle on a different y pane
				if (startX + circleSpace >= width) {
					//calculate borderCorrection for Y
					borderCorrectionY += (radius * (maxborderCorrection + 1))
							* 2 + shapeSize;
					borderCorrectionX = 0;
					//x back to start
					startX = borderCorrectionX + circleSpace;
					maxborderCorrection = 0;
				}
				startY = circleSpace + borderCorrectionY;

				GroupVariables groupVariables = new GroupVariables();
				//add center node
				groupVariables.setFirstNode(firstVisibleNode);

				//add corrected radius of the group
				groupVariables.setRadius((circleLayers + 1)
						* radius - shapeSize);

				if(type.equalsIgnoreCase(ConfidenceSwitch.getConfidenceType())){
					groupName = groupName.split(";")[0];
				}
				//add name of the group
				groupVariables.setGroupName(groupName);
				//add number of nodes in the group
				groupVariables.setNumberOfNodes(nodeCount);
				groupCollection.add(groupVariables);
				//Biggest Circle
				maxborderCorrection = Math.max(maxborderCorrection,
						circleLayers);
				borderCorrectionX = circleSpace;
				//calculate node positions
				calculateNewNodePositions(nodes, startX, startY);
			}
		}
	}
	
	/**
	 * calculates amount of circle layers for one group
	 * @param groupSize
	 * @return amount of circle layers (Math.ceil)
	 */
	protected double calculateCircleLayers(int groupSize){
		return Math.ceil((double) groupSize / layerSize);
	}
	
	/**
	 * calculate circle space (biggest radius of circle)
	 * @param correction
	 * @return circle space (biggest radius)
	 */
	protected double calculateCircleSpace(double correction){
		return radius * (correction + 1) + shapeSize;
	}
	
	/**
	 * Calculates Node Positions from circle center startX, startY
	 * @param nodes list of nodes
	 * @param startX X of center
	 * @param startY Y of center
	 */
	protected void calculateNewNodePositions(ArrayList nodes, double startX, double startY){
		int i = 0;
		double tempStartX = startX;
		double tempStartY = startY;
		double xyArray[][] = calculatePointsOnCircle(radius, startX,
				startY);
		for (Iterator itnodes = nodes.iterator(); itnodes.hasNext();) {
			if (i >= xyArray.length) {
				i = 0;
				radius += RADIUSCONST;
				xyArray = calculatePointsOnCircle(radius, startX,
						startY);
			}
			Node n = (Node)itnodes.next();
			//only set xy if node is visible
			if (n.getEdgeCount() > 0) {
				n.setXY(tempStartX, tempStartY);
				tempStartX = xyArray[i][0];
				tempStartY = xyArray[i][1];
				++i;
			}
		}
	}

	/**
	 * Calculates points on a circle with @param radius and center @param startX and @param startY
	 * @return double array of points on circle
	 */
	protected double[][] calculatePointsOnCircle(int radius, double startX,
			double startY) {
		double s = radius / 1.3;
		double stepSize = s / radius;
		int arraySize = (int) ((2 * Math.PI) / stepSize);
		double xyArray[][] = new double[arraySize][2];
		int i = 0;
		for (double t = 0; t < (2 * Math.PI - stepSize); t = t + stepSize) {
			double Xcirc = radius * Math.cos(t);
			double Ycirc = radius * Math.sin(t);
			xyArray[i][0] = Math.round(startX + Xcirc);
			xyArray[i][1] = Math.round(startY + Ycirc);
			++i;
		}
		return xyArray;
	}

	/**
	 * scales the radius of all circles in the group Collection
	 */
	public void scaleRadius(double scale) {
		for (Iterator iter = groupCollection.iterator(); iter.hasNext();) {
			GroupVariables groupVariables = (GroupVariables) iter.next();
			double size = groupVariables.getRadius();
			size *= scale;
			groupVariables.setRadius(size);
		}

	}

	/**
	 * Gets the string of a group for a certain x,y pair(for tooltip)
	 * @param x
	 * @param y
	 * @return name of group
	 */
	public String getGroup(int x, int y) {
		String toolTip = null;
		for (Iterator iter = groupCollection.iterator(); iter.hasNext();) {
			GroupVariables groupVariables = (GroupVariables) iter.next();
			Node node = groupVariables.getFirstNode();
			double r = groupVariables.getRadius();
			double dist = (node.getX() - x) * (node.getX() - x)
					+ (node.getY() - y) * (node.getY() - y);
			if ((dist < (r * r))) {
				toolTip = "Group: " + groupVariables.getGroupName();
				return toolTip;
			}
		}
		return toolTip;
	}

	/**
	 * @return groupCollection Iterator
	 */
	public Iterator getGroupCollectionIter() {
		return groupCollection.iterator();
	}

}
