/**
 * MinimumCut.java
 */
package jsquid.algorithm;

import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;

import jsquid.algorithm.data.ConfidenceClusterSettings;
import jsquid.algorithm.data.PairNeighbour;

import medusa.MedusaSettings;
import medusa.graph.Edge;
import medusa.graph.Graph;

/**
 * @author sanjit
 * Minimum Cut class to calculate high confidence clusters.
 * principle: find node with highest confidence (edge). try to connect more edges to this group until
 * score is to low. 
 * find next node with a high confidence which is not already used and do the process again until no more confidence
 * clusters can be found
 */
public class MinimumCut implements IConfidenceCluster {

	
	private HashMap edgeMatrix;	//edgematrix key: node name, value: arraylist of neighbours

	private Graph graph;

	private HashMap group;

	private String maxConfKey = "";	//the key with the max confidence edge
	
	private ConfidenceClusterSettings clusterSettings;

	private final static String confidenceString = "avg conf ";
	
	private final static String noConfGroupString = "undefined";
	
	private int counter;

	public MinimumCut(Graph graph) {
		this.graph = graph;
	}

	public MinimumCut(Graph graph, ConfidenceClusterSettings clusterSettings) {
		this(graph);
		this.clusterSettings = clusterSettings;
	}
	
	/**
	 * creates edge matrix for all single edges
	 */
	public void createSingleEdgeMatrix() {
		ArrayList edges = new ArrayList(graph.getSingleEdges());
		createEdgeMatrix(edges);
	}

	/**
	 * creates a edge matrix for a type of medusasettings
	 * edgematrix consists all node as key and value a pairneighbour(name of edgepartner, confidence)
	 */
	public void createEdgeMatrix(MedusaSettings settings) {
		ArrayList edges = copyEdgesList(settings);
		createEdgeMatrix(edges);
	}
	
	private void createEdgeMatrix(ArrayList edges){
		edgeMatrix = new HashMap();
		float maxConf = 0;
		//Iterate over all edges
		for (int i = 0; i < edges.size(); ++i) {
			Edge e = (Edge) edges.get(i);
			float conf = e.getConf();
			int counter = 1;

			ArrayList neighboursNode1 = new ArrayList();
			ArrayList neighboursNode2 = new ArrayList();

			if (edgeMatrix.containsKey(e.n1)) {
				neighboursNode1 = (ArrayList) edgeMatrix.get(e.n1);
			}
			if (edgeMatrix.containsKey(e.n2)) {
				neighboursNode2 = (ArrayList) edgeMatrix.get(e.n2);
			}
			//find all edges with the same nodes n1 and n2
			for (int j = i + 1; j < edges.size(); ++j) {
				Edge e2 = (Edge) edges.get(j);
				if (e2.contains(e.n1) && e2.contains(e.n2)) {
					conf += e2.getConf();
					counter++;
					edges.remove(j);
					--j;
				}
			}
			//avg conf.
			conf /= counter;
			neighboursNode1.add(new PairNeighbour(e.n2, conf));
			neighboursNode2.add(new PairNeighbour(e.n1, conf));

			//save the best confidence edge
			if (maxConf < conf) {
				maxConfKey = e.n1;
				maxConf = conf;
			}

			edgeMatrix.put(e.n1, neighboursNode1);
			edgeMatrix.put(e.n2, neighboursNode2);
			edges.remove(i);
			i = -1;
		}
	}

	/**
	 * Groups by Confidence
	 */
	public void groupByConfidence() {
		//Arraylist for all used elements in the grouping process
		ArrayList usedElements = new ArrayList();
		group = new HashMap();
		counter = 0;
		while (maxConfKey.length() != 0) {
			createGroup(usedElements);
			++counter;
		}
		finalizeGroup(usedElements);
	}

	/**
	 * finalize Group. All Elements which are not in a group now will be grouped now. Every
	 * node is one group with no confidence
	 * @param usedElements Elements which are grouped
	 */
	private void finalizeGroup(ArrayList usedElements) {
		int i = 1;
		for (Iterator iternodes = graph.getNodeHM().keySet().iterator(); iternodes.hasNext();) {
			String keyName = (String)iternodes.next();
			if(!usedElements.contains(keyName)){
				ArrayList temp = new ArrayList();
				temp.add(graph.getNode(keyName));
				group.put(noConfGroupString + i, temp);
				++i;
				usedElements.add(keyName);
			}
		}
		
	}

	/**
	 * creates the confidence groups.
	 * finds the edge with the maximum confidence. try to find other edges which can be connected to 
	 * the group (they have to satisfy confidenceCut and connectionCut). If no more edges can be connected it will 
	 * try to find the next possible group.
	 * @param usedElements
	 */
	private void createGroup(ArrayList usedElements) {

		int connections = 0; //number of connections in a group
		float groupConfidence = 0; //sum of group confidence
		float averageConfidence = 0;	//avg confidence of group
		//gets the neighbourslist from the key with the maximum confidence
		ArrayList maximumConfList = (ArrayList) edgeMatrix.get(maxConfKey);
		ArrayList groupList = new ArrayList(); //list of elements which belong to the group
		groupList.add(graph.getNode(maxConfKey));
		usedElements.add(maxConfKey);

		Collections.sort(maximumConfList); // Sorted by Confidence
		PairNeighbour first = (PairNeighbour) maximumConfList.remove(0);
		//try to find new Neighbour which is not used
		while(usedElements.contains(first.getNodeName()) && !maximumConfList.isEmpty()){
			first = (PairNeighbour) maximumConfList.remove(0);
		}
		//if Neighbour does not satisfy confidenceCutoff create no conf group
		if (first.getConfidence() < clusterSettings.getConfidenceCut() || usedElements.contains(first.getNodeName())) {
			group.put(noConfGroupString + 0, groupList);
			maxConfKey = "";
			return;
		}

		connections++;
		//edgeMatrix.put(maxConfKey, maximumConfList);
		//Merge lists of edges between orginal node and edge node (Minimum Cut princip)
		maxConfKey = mergeLists(maxConfKey, first.getNodeName());
		groupList.add(graph.getNode(first.getNodeName()));
		usedElements.add(first.getNodeName());
		//Calculate Average confidence of the group
		averageConfidence = groupConfidence = first.getConfidence();
		//maximumConfList = (ArrayList) edgeMatrix.get(maxConfKey);
		
		//Iterate through all neighbours in the list
		for (Iterator iterConfList = maximumConfList.iterator(); iterConfList
				.hasNext();) {
			PairNeighbour neighbour = (PairNeighbour) iterConfList.next();
			//neighbour should not be member of a diffrent group or already used
			if (!usedElements.contains(neighbour.getNodeName())) {
				//Get all edges to the group
				ArrayList edgesToGroupList = getAllConnections(neighbour.getNodeName(), maximumConfList);
				int toGroupConnections = edgesToGroupList.size();
				float toGroupConfidence = 0;
				//Iterate through all edges to the group and calculate averageConfidence
				for (Iterator iter = edgesToGroupList.iterator(); iter
						.hasNext();) {
					toGroupConfidence += ((PairNeighbour) iter.next()).getConfidence();
				}
				
				// maxConnection /= groupList.size();
				averageConfidence = groupConfidence / connections;
				//maxConnection = groupList.size();
				float toGroupAvgConfidence = toGroupConfidence
						/ toGroupConnections;
				//Check the score. Comparsion between ratio connections and ratio confidence
				if (checkScore((float) (toGroupConnections / (float)groupList.size()),
						toGroupAvgConfidence)) {
					//add to group
					groupConfidence += toGroupConfidence;
					connections += toGroupConnections;
					groupList.add(graph.getNode(neighbour.getNodeName()));
					usedElements.add(neighbour.getNodeName());
					maxConfKey = mergeLists(maxConfKey, neighbour.getNodeName());
					averageConfidence = groupConfidence / connections;
					//maximumConfList = (ArrayList) edgeMatrix.get(maxConfKey);
				}
				iterConfList = maximumConfList.iterator();
			}
			else iterConfList.remove(); //remove neighbour from list
		}

		// averageConfidence = compConfidence / groupList.size();
		//add group
		DecimalFormat df = new DecimalFormat("0.00");
		group.put(confidenceString + df.format(averageConfidence) + ";" + counter, groupList);
		//remove key
		edgeMatrix.remove(maxConfKey);
		//find next node which a high confidence
		findNextMaxConf(usedElements);
	}

	/**
	 * Get all connection in the maximumConfList to the node (nodeName)
	 * @param nodeName
	 * @param maximumConfList
	 * @return List of all connections to the node (nodeName)
	 */
	private ArrayList getAllConnections(String nodeName, ArrayList maximumConfList) {
		ArrayList edges = new ArrayList();
		for (Iterator iter = maximumConfList.iterator(); iter.hasNext();) {
			PairNeighbour node = (PairNeighbour)iter.next();
			if(node.getNodeName().equals(nodeName)){
				edges.add(node);
				iter.remove();
			}
			
		}
		return edges;
	}

	/**
	 * Merges two lists of PairNeighbours. Deletes old keys in edgeMatrix. Creates new key which is node1;node2
	 * @param node1 name of node1
	 * @param node2 name of node2
	 * @return
	 */
	private String mergeLists(String node1, String node2) {
		ArrayList list1 = (ArrayList) edgeMatrix.get(node1);
		ArrayList list2 = (ArrayList) edgeMatrix.get(node2);
		list1.addAll(list2);
		Collections.sort(list1);
		edgeMatrix.remove(node1);
		edgeMatrix.remove(node2);
		node1 += ";" + node2;
		edgeMatrix.put(node1, list1);
		return node1;
	}

	/**
	 * tries to find the next node with a confidence higher then the confidenceCut.
	 * elements which are in usedelements will be excluded from search
	 * @param usedElements
	 */
	private void findNextMaxConf(ArrayList usedElements) {
		PairNeighbour maximumNode = new PairNeighbour("", 0);
		for (Iterator iter = edgeMatrix.keySet().iterator(); iter.hasNext();) {
			String keyName = (String) iter.next();
			if (!usedElements.contains(keyName)) {
				ArrayList neighbourList = (ArrayList)edgeMatrix.get(keyName);
				Collections.sort(neighbourList);
				for (Iterator iterList = neighbourList.iterator(); iterList
						.hasNext();) {
					PairNeighbour pair = (PairNeighbour) iterList.next();
					if (!usedElements.contains(pair.getNodeName())) {
						if (pair.getConfidence() > clusterSettings.getConfidenceCut())
							maximumNode = pair.getConfidence() > maximumNode
									.getConfidence() ? pair : maximumNode;
					}
				}
			}
		}
		maxConfKey = maximumNode.getNodeName();

	}

	/**
	 * checks the score (ratio connections to group/connections in group and ratio confidence average confidence
	 * from new node to group/average confidence in group)
	 */
	private boolean checkScore(float i, float f) {
		boolean goodScore = true;
		if (i < clusterSettings.getConnectionCut() || f < clusterSettings.getConfidenceCut())
			goodScore = false;
		return goodScore;
	}

	/**
	 * copies all edges which belong to a specific setting to a new list
	 * @param settings
	 * @return new list
	 */
	private ArrayList copyEdgesList(MedusaSettings settings) {
		ArrayList edges = new ArrayList();
		for (Iterator edgeIter = graph.edgesIterator(); edgeIter.hasNext();) {
			Edge e = (Edge) edgeIter.next();
			if (settings.getColor(new Integer(e.getType())) != null)
				edges.add(e);
		}
		return edges;
	}

	/**
	 * @return group
	 */
	public HashMap getGroup() {
		return group;
	}


}
