/**
 * JSquidPanel.java
 *
 */

package jsquid;

import java.awt.AlphaComposite;
import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Composite;
import java.awt.Font;
import java.awt.Graphics2D;
import java.awt.Paint;
import java.awt.Point;
import java.awt.Shape;
import java.awt.Stroke;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseEvent;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.Set;

import javax.swing.JColorChooser;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPopupMenu;
import javax.swing.Timer;

import jsquid.display.GroupDisplay;
import jsquid.display.GroupVariables;
import jsquid.display.JLinkButton;
import jsquid.display.LocationGroupDisplay;
import jsquid.display.OrganelleType;
import jsquid.display.OverviewPanel;
import jsquid.display.ShapeDialog;
// import jsquid.display.OverviewPanel2;
import jsquid.display.drawing.PaintStyle;
import jsquid.display.drawing.factory.DrawingFactory;
import jsquid.display.drawing.factory.DrawingType;
import jsquid.graph.JSquidGraph;
import jsquid.xml.LinkURL;
import jsquid.xml.XMLDataLoader;
import medusa.MedusaSettings;
import medusa.display.BasicGraphPanel;
import medusa.display.PaintTools;
import medusa.graph.Edge;
import medusa.graph.Node;

/**
 * @author sanjit and martin
 * 
 * the main panel which contains the graph and offers several modification functionalities
 * The class extends Medusa's BasicGraphPanel which is the basis of the whole Applet. The 
 * BasicGraphPanel has also been modified to some extend but the key features have been kept the same,
 * since Medusas turned out to be quite stable and a good basis for further expansions.
 * 
 */
public class JSquidPanel extends BasicGraphPanel implements ActionListener {

	//contains a Boolean value for each interaction type (e.g. physical interaction) indicating 
	//if edges of these types should be drawn or not
	boolean[] showEdges;

	//holds the color table, checked table and interaction table
	MedusaSettings settings = null;
	
	//Medusa settings for special interaction types like orthologs
	//or paralogs which do not a have a confidence and therefore have
	//to be treated differently
	MedusaSettings nonConfSettings = null;

	//needed for grouping
	GroupDisplay grDisp;

	//represents the reference to the JSquidApplet. This is needed e.g. for hyper linking
	private JSquidApplet appletRef = null;

	//represents the previous mouse position within the panel - needed for some muse events
	private Point posAtPrevCall = null;

	//this popup appears on right-clicking a node - it contains hyper links, synonyms, features
	// for the distinct protein (node)
	private JPopupMenu nodePopup = new JPopupMenu("Lookup");

	//reference to the overview panel - needed to pass correction values to the overview panel
	private OverviewPanel overviewRef = null;

	//this factory class instanciates the proper drawing style (e.g. Mednusa Style or JSquid Style) 
	private DrawingFactory factory;

	//the current drawing style(e.g. Mednusa Style or JSquid Style) 
	private PaintStyle paintStyle;

	//the current grouping type (e.g. "Location" or "Confidence")
	private String groupType = "";

	//Buttens that appear in the nodePopup to expand/collapse nodes or change shape/color of nodes 
	private JLinkButton collapsButton = null;
	private JLinkButton colorButton = null;
	private JLinkButton shapeButton = null;
	private JLinkButton expandButton = null;
	private JLinkButton renameCollNodeButton = null;

	//the previously richt-clicked node
	private Node lastClickedNode = null;

	//the list of edges which are supposed to be shown (not all edges are supposed to shown all the time
	// e.g. some are not shown, when the confidence range is limited)
	private ArrayList subEdges = null;

	//indicates whether edge highlighting on mouse-over-node is active
	private boolean areEdgesHighlighted = false;
	
	//indicates whether the popup is used (mouse is over popup)
	private boolean isPopupUsed = false;
	
	//used to disable the popup after some time, if not used
	private Timer popupTimer = null;

	//the minimum confidence for shown edges
	private int lowValue = 0;

	//the maximum confidence for shown edges
	private int highValue = 10;

	//counts the collapsed nodes (nodes which are reduced to one node, containig all their edges)
	private int collapsNodesCounter = 0;

	//representing the Node which is about to be expanded again
	private String toExpandNode = "";
	
	//the number of all interaction types (species + types + predictions + non-confidence)
	int edgeCount;
	
	//determines the number of non-confidence type 
	private int nonConfSize = 0;


	public JSquidPanel(MedusaSettings ms, JSquidApplet appletRef,
			OverviewPanel overviewRef, int edgeCount) {
		super(ms);
		this.settings = ms;
		this.appletRef = appletRef;
		this.overviewRef = overviewRef;
		factory = new DrawingFactory();
		paintStyle = factory.create(DrawingType.NORMALTYPE);
		this.edgeCount = edgeCount;
		initShow(edgeCount);
		this.addMouseMotionListener(this);
	}
	
	private void initNonConfSize() {
		Hashtable nonConfCheckedTable = null;
		if (nonConfSettings != null)
			nonConfCheckedTable = nonConfSettings.getCheckedTable();
		if (nonConfCheckedTable != null)
			nonConfSize = nonConfCheckedTable.size();
	}
	
	public String getGroupType() {
		return this.groupType;
	}
	
	public void setShowGroupBorders(boolean showGroupBorders) {
		this.showGroupBorders = showGroupBorders;
	}

	/**
	 * initializes the array representing the visibility of edges of certain interaction types
	 */
	private void initShow(int size) {
		showEdges = new boolean[size];
		for (int i = 0; i < size; i++)
			showEdges[i] = false;
		
		//init nonConf types - that should happen only once
		Hashtable nonConfCheckedTable = null;
		if (nonConfSettings != null)
			nonConfCheckedTable = nonConfSettings.getCheckedTable();
		for (int i = 0; i < showEdges.length; ++i) {
			if (nonConfCheckedTable != null && nonConfCheckedTable.containsKey(new Integer(i+1)))
				showEdges[i] = ((Boolean) nonConfCheckedTable.get(new Integer(i+1)))
						.booleanValue();
		}
	}

	/**
	 * setting all values in the array to false
	 * 
	 */
	
	private void disableShow() {
		for (int i = 0; i < showEdges.length-nonConfSize; ++i) {
			showEdges[i] = false;
		}
	}
	
	/**
	 * 
	 * @param i
	 * @param b
	 * 
	 * sets one value in the array
	 */
	public void setShowEdges(int i, boolean b) {
		showEdges[i - 1] = b;
	}


	/**
	 * 
	 * updates the visibility of edges of certain interaction types according to the current
	 * MeduasSettings object which contains inforamtion about this visibility
	 */
	public void updateShow() {
		disableShow();
		Hashtable checkedTable = settings.getCheckedTable();
		for (int i = 0; i < showEdges.length - nonConfSize; ++i) {
			if (checkedTable.containsKey(new Integer(i + 1)))
				showEdges[i] = ((Boolean) checkedTable.get(new Integer(i + 1)))
						.booleanValue();
			else
				showEdges[i] = false;
		}
	}

	public void setSettings(MedusaSettings settings) {
		this.settings = settings;
	}
	
	public void setNonConfSettings(MedusaSettings settings) {
		this.nonConfSettings = settings;
		initNonConfSize();
		initShow(edgeCount);
	}

	public MedusaSettings getSettings() {
		return this.settings;
	}

	public void initSubEdgesList() {
		subEdges = null;
	}
	
	/**
	 * overrides the method from BasicGraphPanel.
	 * This method draws Nodes, Edges and grouping circles, if grouping has been activated
	 */
	public void paintNet(Graphics2D g2d) {
		//decides whether to display the simple or the extended graph
		if (subEdges == null && prettyEdge)
			subEdges = graph.getEdges();
		else if (subEdges == null)
			subEdges = graph.getSingleEdges();
		
		paintStyle.setProperties(graph, showConfidence, prettyNode, prettyEdge,
				labelNode, labelEdge, uniformNodes, settings, nonConfSettings);
		
		//set edge count for all nodes to zero
		initEdgeCount();
		
		//if the graph is the extended graph
		if (prettyEdge) {
			//paint extended graph edges
			for (Iterator edges = subEdges.iterator(); edges.hasNext();) {
				Edge e = (Edge) edges.next();
				paintEdge(g2d, e);
			}
		}
		else {
			//paint simple graph edges
			for (Iterator itor = subEdges.iterator(); itor.hasNext();) {
				Edge e = (Edge) itor.next();
				graph.getNode(e.getFromName()).increaseEdgeCount();
				graph.getNode(e.getToName()).increaseEdgeCount();
				paintStyle.paintEdge(g2d, e);
			}
		}
		
		//if grouping is active
		if (showGroupBorders)
			paintGroupBorders(g2d);
		
		g2d.setComposite(makeComposite(1.0f));
		//paint nodes
		for (Iterator it = graph.nodesIterator(); it.hasNext();) {
			Node node = (Node) it.next();
			if (node.getEdgeCount() > 0 || node.isCollapseNode()) {
				node.setVisible(true);
				paintNode(g2d, node);
			}
			else
				node.setVisible(false);
		}
	}
	
	/**
	 * 
	 * @param g2d
	 * @param shape
	 * @param color
	 * 
	 * draws one of the biologically shaped organelles used for subcellular location grouping
	 */
	private void drawOrganelle(Graphics2D g2d, Shape shape, Color color) {
		Composite orginalComp = g2d.getComposite();
		Paint originalPaint = g2d.getPaint();
		Stroke originalStroke = g2d.getStroke();
		AlphaComposite ac = AlphaComposite.getInstance(AlphaComposite.SRC_OVER,
				0.5f);
		
		g2d.setComposite(ac);
		g2d.setPaint(color);
		g2d.setStroke(new BasicStroke(5.0f));
		g2d.draw(shape);
		
		g2d.setPaint(originalPaint);
		g2d.setStroke(originalStroke);
		g2d.setComposite(orginalComp);
	}

	//sets the edge cout for every node to 0
	private void initEdgeCount() {
		for (Iterator it = graph.nodesIterator(); it.hasNext();) {
			Node node = (Node) it.next();
			node.setEdgeCount(0);
		}
	}
	
	/**
	 * method for painting the edges of the extended graph
	 */
	public void paintEdge(Graphics2D g, Edge e) {

		// g.setPaint(Color.blue);
		try {
			if (!showEdges[e.getType() - 1])
				return;
			graph.getNode(e.getFromName()).increaseEdgeCount();
			graph.getNode(e.getToName()).increaseEdgeCount();
			paintStyle.paintEdge(g, e);
		} catch (IndexOutOfBoundsException ex) {
			//do nothing
		}
	}

	/**
	 * method for painting nodes
	 */
	public void paintNode(Graphics2D g, Node n) {
		paintStyle.paintNode(g, n);
		createNodePopup(g);
	}

	/**
	 * 
	 * @param g2d
	 * 
	 * paints the group borders when grouping has been activated. By default, circles are draw around
	 * the different groups. Though, there are special shapes for the subcellular location grouping
	 */
	public void paintGroupBorders(Graphics2D g2d) {
		Composite orginal = g2d.getComposite();
		Font originalFont = g2d.getFont();
		AlphaComposite ac = AlphaComposite.getInstance(AlphaComposite.SRC_OVER,
				0.5f);
		g2d.setComposite(ac);
		g2d.setColor(new Color(70,130,180));
		//iterate over all groups of the currently selected grouping type
		for (Iterator iter = grDisp.getGroupCollectionIter(); iter.hasNext();) {
			//getting the information about the groups
			GroupVariables groupVariables = (GroupVariables) iter.next();
			int r = (int) groupVariables.getRadius();
			int size = r * 2;
			Node n = groupVariables.getFirstNode();
			
			//paint borders only if there are nodes assigned to this group
			if (groupVariables.getNumberOfNodes() > 0) {
				String groupName = groupVariables.getGroupName();
				
				//special shapes for subcellular location
				if (groupType.equalsIgnoreCase(XMLDataLoader.LOCATION_GRP_CONST)) {
					if (groupName.equalsIgnoreCase(OrganelleType.ORGANELLE_MITO))
						drawOrganelle(g2d, PaintTools.mito((int)n.getX(), (int)n.getY(), size), new Color(0,153,0));
					else if (groupName.equalsIgnoreCase(OrganelleType.ORGANELLE_ER))
						drawOrganelle(g2d, PaintTools.ER((int)n.getX(), (int)n.getY(), size), new Color(200,8,21));
					else if (groupName.equalsIgnoreCase(OrganelleType.ORGANELLE_GOLGI))
						drawOrganelle(g2d, PaintTools.golgi((int)n.getX(), (int)n.getY(), size), new Color(10,130,180));
					else if (groupName.equalsIgnoreCase(OrganelleType.ORGANELLE_NUCLEUS))
						drawOrganelle(g2d, PaintTools.nucleus((int)n.getX(), (int)n.getY(), size), new Color(100,100,100));
					else if (groupName.equalsIgnoreCase(OrganelleType.ORGANELLE_MEMBRANE))
						drawGroupingCircle(g2d, (int)n.getX() - size, (int)n.getY() - r, size, 5.0f);
					else if (groupName.equalsIgnoreCase(OrganelleType.ORGANELLE_EXTRACELL))
						drawGroupingRectangle(g2d,(int)n.getX() - r, (int)n.getY() - r, size, new Color(255,117,24));		
					else if (groupName.equals(OrganelleType.ORGANELLE_UNDEF))
						drawGroupingCircle(g2d, (int)n.getX() - r, (int)n.getY() - r, size, 3.0f);
					//no group borders for cytoplasm needed!
				}
				
				//draw normal grouping circles around the groups
				else if (groupVariables.getNumberOfNodes() > 1){
					drawGroupingCircle(g2d, (int)n.getX() - r, (int)n.getY() - r, size, 3.0f);
				}
				
				if (labelGroup && (groupVariables.getNumberOfNodes() > 1) 
						|| groupType.equalsIgnoreCase(XMLDataLoader.LOCATION_GRP_CONST)){
					// draw group name
					g2d.setComposite(AlphaComposite.getInstance(AlphaComposite.SRC_OVER, 1.0f));
					g2d.setFont(new Font("SansSerif", Font.BOLD, 12));
					int grpNameLen = groupName.length();
					int nameDrawCorrX = grpNameLen / 2 * 6; // 6 is average letter width
					int nameDrawCorrY = 0;
					
					//special correction subcellular grouping (due to the shapes are not circular)
					if (groupType.equalsIgnoreCase(XMLDataLoader.LOCATION_GRP_CONST)) {
						if (groupName.equalsIgnoreCase(OrganelleType.ORGANELLE_MEMBRANE)) {
							nameDrawCorrX += r;
							nameDrawCorrY = -r/10;
						}
						else if (groupName.equalsIgnoreCase(OrganelleType.ORGANELLE_ER) ||
								groupName.equalsIgnoreCase(OrganelleType.ORGANELLE_MITO))
							nameDrawCorrY = r/5;
						else if (groupName.equalsIgnoreCase(OrganelleType.ORGANELLE_GOLGI))
							nameDrawCorrY = r/3;
					}
					g2d.drawString(groupName, (int) n.getX() - nameDrawCorrX,
							(int) n.getY() - r - 5 + nameDrawCorrY);
					g2d.setComposite(ac);
				}
			}
		}
		
		g2d.setComposite(orginal);
		g2d.setFont(originalFont);
	}
	
	/**
	 * 
	 * @param g2d
	 * @param x
	 * @param y
	 * @param size
	 * @param stroke
	 * 
	 * draws the standard grouping circle around the groups
	 */
	private void drawGroupingCircle(Graphics2D g2d, int x, int y, int size, float stroke) {
		Stroke originalStroke = g2d.getStroke();
		g2d.setStroke(new BasicStroke(stroke));
		g2d.drawOval(x, y, size, size);
		g2d.setStroke(originalStroke);
	}
	
	/**
	 * 
	 * @param g2d
	 * @param x
	 * @param y
	 * @param size
	 * @param col
	 * 
	 * draws a rounded rectangel around the group (currently used for extracellular group within
	 * subcell. location grouping
	 */
	private void drawGroupingRectangle(Graphics2D g2d, int x, int y, int size, Color col) {
		size = (int) (size * 0.9);
		Stroke originalStroke = g2d.getStroke();
		Paint originalPaint = g2d.getPaint();
		g2d.setStroke(new BasicStroke(5.0f));
		g2d.setPaint(col);
		g2d.drawRoundRect(x, y, size, size, size/5, size/5);
		g2d.setStroke(originalStroke);
		g2d.setPaint(originalPaint);
	}

	//creates the popup which appears on right-clicking a node
	private void createNodePopup(Graphics2D graphics) {
		Node node = getRightClickedNode();
		if (node != null) {
			lastClickedNode = node;
			nodePopup.setVisible(false);

			nodePopup.removeAll();
			//add the options to the popup
			nodePopup.add(new JLabel("<html><b>Options:</b></html>"));
			ArrayList fixedNodes = graph.getFixed();
			//add expand and rename option, if the selected node has been collapesed
			if (node.isCollapseNode() && fixedNodes.size() <= 1) {
				expandButton = new JLinkButton("expand");
				renameCollNodeButton = new JLinkButton("rename");
				nodePopup.add(expandButton);
				nodePopup.add(renameCollNodeButton);
				toExpandNode = node.getLabel();
				expandButton.addActionListener(this);
				renameCollNodeButton.addActionListener(this);
			}
			//add collapse option, if selection affects more than 1 node
			if (node.isFixed() && fixedNodes.size() > 1) {
				collapsButton = new JLinkButton("collapse");
				nodePopup.add(collapsButton);
				collapsButton.addActionListener(this);
			}
			
			//add color option
			colorButton = new JLinkButton("color");
			nodePopup.add(colorButton);
			colorButton.addActionListener(this);
			
			//add shape option
			shapeButton = new JLinkButton("shape");
			nodePopup.add(shapeButton);
			shapeButton.addActionListener(this);
			
			//add additional iformation, if the selected node is not a collapsed one
			if (!node.isCollapseNode()) {
				nodePopup.addSeparator();
				
				//add hyper links
				nodePopup.add(new JLabel("<html><b>Lookup:</b></html>"));
				HashMap linkStubMap = graph.getLinkStubMap();
				HashMap linkIDMap = node.getLinkIDMap();

				if (linkStubMap != null && linkIDMap != null
						&& linkStubMap.size() > 0 && linkIDMap.size() > 0) {
					Set linkIDTable = linkIDMap.keySet();
					for (Iterator itor = linkIDTable.iterator(); itor.hasNext();) {
						String id = (String) itor.next();
						if (linkStubMap.containsKey(id)) {
							ArrayList linkIDList = (ArrayList) linkIDMap
									.get(id);
							for (int i = 0; i < linkIDList.size(); ++i) {
								LinkURL url = (LinkURL) linkStubMap.get(id);
								url.setLinkID((String) linkIDList.get(i));
								String urlString = url.toString();

								JLinkButton lButton = null;
								try {
//									lButton = new JLinkButton(JLinkButton.makePrettyURLName(urlString),
//											new URL(urlString));
									lButton = new JLinkButton(url.getLinkName(),
											new URL(urlString));
								} catch (MalformedURLException e) {
									// do nothing
								}
								nodePopup.add(lButton);
								lButton.addActionListener(this);
							}
						}
					}
				} else {
					try {
						JLinkButton lButton = new JLinkButton(
								"No links specified", new URL(
										"http://www.google.com/search?q="
												+ node.getLabel()));
						nodePopup.add(lButton);
						lButton.addActionListener(this);
					} catch (MalformedURLException e) {
						// do nothing
					}
				}
			}

			//add synonyms
			ArrayList syns = node.getSynonyms();
			if (syns != null && syns.size() > 0) {
				nodePopup.addSeparator();
				nodePopup.add(new JLabel("<html><b>Synonyms:</b></html>"));
				for (int i = 0; i < syns.size(); ++i) {
					JLabel synLabel = new JLabel(
							"<html>&nbsp;&nbsp;&nbsp;&nbsp;"
									+ (String) syns.get(i) + "</html>");
					nodePopup.add(synLabel);
				}
			}
			
			//add features (the groups this node belongs to)
			ArrayList groups = node.getGroups();
			if (groups != null && groups.size() > 0) {
				nodePopup.addSeparator();
				nodePopup.add(new JLabel("<html><b>Features:</b></html>"));
				for (int i = 0; i < groups.size(); ++i) {
					JLabel grpLabel = new JLabel(
							"<html>&nbsp;&nbsp;&nbsp;&nbsp;"
									+ (String) groups.get(i) + "</html>");
					nodePopup.add(grpLabel);
				}
			}

			nodePopup.addMouseListener(this);

			Point point = this.getLocationOnScreen();
			nodePopup.setLocation((int) node.getX() + point.x + 10, (int) node
					.getY()
					+ point.y + 10);

			nodePopup.setVisible(true);
			setRightClickedNode(null);
			
			if (popupTimer == null)
				popupTimer = new Timer(5000, setPopupInvisible);
			popupTimer.restart();
		}
	}
	
	private ActionListener setPopupInvisible = new ActionListener() {
		public void actionPerformed(ActionEvent evt) {
			if (nodePopup != null && !isPopupUsed) {
				popupTimer.stop();
				nodePopup.setVisible(false);
			}
		}
	};

	/**
	 * 
	 * @param type
	 * 
	 * groups the nodes according to their group membership
	 */
	public void groupBy(String type) {
		//due to a better visibility the drawing type is switched to the Medusa style
		switchStyle(DrawingType.OLDTYPE);
		appletRef.switchStyleButton(DrawingType.OLDTYPE);
		
		if (type.equalsIgnoreCase(XMLDataLoader.LOCATION_GRP_CONST))
			grDisp = new LocationGroupDisplay(getPanelWidth(), getPanelHeight(), graph);
		else
			grDisp = new GroupDisplay(getPanelWidth(), graph);
		grDisp.groupNodes(type);
		groupType = type;
		showGroupBorders = true;
	}

	public void mouseClicked(MouseEvent e) {
		super.mouseClicked(e);
		
		//if a click outside the nodePopup occurs, it will be hidden
		if (!nodePopup.getBounds().contains(e.getPoint()))
			nodePopup.setVisible(false);

	}

	/**
	 * 
	 * @param scale
	 * 
	 * scales the graph. The node which is the closest one to the center is used as a reference node.
	 * This is important to prevent the graph from "jumping around" during scaling
	 */
	public void rescaleNodes(double scale) {
		Collection nodeTable = graph.getNodes().values();
		
		double correctionX = 0.0;
		double correctionY = 0.0;
		Node centerNode = null;
		Point lastMouseWheelPos = appletRef.getLastPosOnMouseWheelScroll();
		if (lastMouseWheelPos == null)
			centerNode = getClosestToCenter();
		else 
			centerNode = getClosestToMouse(lastMouseWheelPos);
		
		if (centerNode != null) {
			double prevXPos = centerNode.getX();
			double prevYPos = centerNode.getY();
			
			centerNode.rescale(scale);
			
			double actualXPos = centerNode.getX();
			double actualYPos = centerNode.getY();
		
			//the center node will stay at the same position during scaling
			//prevents the garph from "jumping"
			correctionX = prevXPos - actualXPos;
			correctionY = prevYPos - actualYPos;
			
			centerNode.move(correctionX, correctionY);
		}
		
		//scale all nodes
		for (Iterator itor = nodeTable.iterator(); itor.hasNext();) {
			Node n = (Node) itor.next();
			if (n != null && n!= centerNode) {
				n.rescale(scale);
				//correction according to the center node
				n.move(correctionX, correctionY);
			}
		}
		
		//scale biologically shaped subcell. location group borders that do not contain any nodes.
		//we decided to also show empty subcell. location groups to make the cell look more recognizable
		ArrayList emptyOrganelleGroupNodes = ((JSquidGraph)graph).getEmptyOrganelleGroupNodes();
		if (emptyOrganelleGroupNodes != null) {
			for (Iterator itor = emptyOrganelleGroupNodes.iterator(); itor.hasNext();) {
				Node n = (Node) itor.next();
				if (n != null && n!= centerNode) {
					n.rescale(scale);
					n.move(correctionX, correctionY);
				}
			}
		}
		
		//scales the standard grouping cycle
		if (grDisp != null)
			grDisp.scaleRadius(scale);

		overviewRef.resizeOverviewRect(scale);

		repaint();
	}
	
	/**
	 * 
	 * @return
	 * returns the node which is the closest one to the center
	 */
	private Node getClosestToCenter() {
		Node bestNode = null;
		double bestdist = Double.MAX_VALUE;
		int x = getWidth()/2;
		int y = getHeight()/2;

		for (Iterator it = graph.nodesIterator(); it.hasNext();) {
			Node node = (Node) it.next();
			double dist = (node.getX() - x) * (node.getX() - x)
					+ (node.getY() - y) * (node.getY() - y);

			if (dist < bestdist) {
				bestdist = dist;
				bestNode = node;
			}
		}
		return bestNode;
	}
	
	/**
	 * 
	 * @param point
	 * @return
	 * returns the node which is the closest one to the mouse position - 
	 * if the scrollwheel is moved when the mouse is over the JSquidPanel
	 */
	private Node getClosestToMouse(Point point) {
		Node bestNode = null;
		double bestdist = Double.MAX_VALUE;
		int x = point.x;
		int y = point.y;

		for (Iterator it = graph.nodesIterator(); it.hasNext();) {
			Node node = (Node) it.next();
			double dist = (node.getX() - x) * (node.getX() - x)
					+ (node.getY() - y) * (node.getY() - y);

			if (dist < bestdist) {
				bestdist = dist;
				bestNode = node;
			}
		}
		return bestNode;
	}
	
	public void mouseDragged(MouseEvent e) {
		super.mouseDragged(e);

		// right button drag or middle button drag
		if (((e.getModifiers() & MouseEvent.BUTTON3_MASK) == MouseEvent.BUTTON3_MASK)
				|| ((e.getModifiers() & MouseEvent.BUTTON2_MASK) == MouseEvent.BUTTON2_MASK)){
			if (posAtPrevCall == null)
				posAtPrevCall = e.getPoint();

			Point actualPos = e.getPoint();
			double xDiff = actualPos.getX() - posAtPrevCall.getX();
			double yDiff = actualPos.getY() - posAtPrevCall.getY();

			moveNodes(xDiff, yDiff);

			overviewRef.repaint();

			posAtPrevCall = actualPos;
		}
	}

	public void mouseReleased(MouseEvent e) {
		super.mouseReleased(e);

		posAtPrevCall = null;
	}

	/**
	 * 
	 * @param xMove
	 * @param yMove
	 * 
	 * moves all nodes in the JSquidPanel
	 */
	public void moveNodes(double xMove, double yMove) {
		Collection nodeTable = graph.getNodes().values();
		for (Iterator itor = nodeTable.iterator(); itor.hasNext();) {
			Node n = (Node) itor.next();
			if (n != null)
				n.move(xMove, yMove);
		}
		//move biologically shaped subcell. location group borders that do not contain any nodes.
		//we decided to also show empty subcell. location groups to make the cell look more recognizable
		ArrayList emptyOrganelleGroupNodes = ((JSquidGraph)graph).getEmptyOrganelleGroupNodes();
		if (emptyOrganelleGroupNodes != null) {
			for (Iterator itor = emptyOrganelleGroupNodes.iterator(); itor.hasNext();) {
				Node n = (Node) itor.next();
				if (n != null)
					n.move(xMove, yMove);
			}
		}
		repaint();
	}

	//returns the tooltip text of a node (annotation), if the mouse is on a node
	public String getToolTipText(MouseEvent e) {
		
		String toolTip = super.getToolTipText(e);
		if (toolTip == null && showGroupBorders && grDisp != null) {
			toolTip = grDisp.getGroup(e.getX(), e.getY());
		}
		if(toolTip != null)
			appletRef.performTooltipInvokeAction();
		return toolTip;

	}

	protected void repaintOverviewPanel() {
		overviewRef.repaint();
	}

	public void switchStyle(String type) {
		paintStyle = factory.create(type);

		if (paintStyle == null) {
			type = DrawingType.NORMALTYPE;
			paintStyle = factory.create(type);
		}
		
		repaint();
	}

	public void actionPerformed(ActionEvent e) {
		try {
			if (e.getSource() instanceof JLinkButton) {
				nodePopup.setVisible(false);	
			}
			JLinkButton sender = (JLinkButton) e.getSource();
			//collapse button event
			if (sender == collapsButton) {
				showGroupBorders = false;
				((JSquidGraph)graph).collapseNodes(prettyEdge);
				graph.disableEdgeHighlighting();
				setEdgeRange(lowValue, highValue);
				++collapsNodesCounter;
				if (appletRef != null)
					appletRef.switchVisibilityGroupButton(false);
			} 
			//expand button event
			else if (sender == expandButton) {
				try {
					showGroupBorders = false;
					((JSquidGraph)graph).expandNode(toExpandNode);
					graph.disableEdgeHighlighting();
					setEdgeRange(lowValue, highValue);
					--collapsNodesCounter;
					if (appletRef != null) {
						appletRef.getSearchField().changedUpdate(null);
						if(collapsNodesCounter == 0)
							appletRef.switchVisibilityGroupButton(true);
					}
				} catch (Exception ex) {
					System.out.println(ex.getMessage());
				}
			} 
			//rename button event
			else if (sender == renameCollNodeButton) {
				Node node = graph.getNode(toExpandNode);
				String input = (String) JOptionPane.showInputDialog(this,
						"Enter New Name for Collapsed Node", "Rename Collapsed Node",
						JOptionPane.QUESTION_MESSAGE, null, null, node.getDisplayLabel());
				input += " ( collapsed )";
				node.setDisplayLabel(input);
			}
			// color button event
			else if (sender == colorButton) {
				Color newColor = JColorChooser.showDialog(
	                     this, "Choose node color", lastClickedNode.getColor());
				if (newColor != null) {
					lastClickedNode.setColor(newColor);
					
					ArrayList fixedNodes = graph.getFixed();
					if (fixedNodes.size() > 0 && fixedNodes.contains(lastClickedNode)) {
						for (int i = 0; i < fixedNodes.size(); ++i) 
							((Node)(fixedNodes.get(i))).setColor(newColor);
					}
				}
			}
			//shape button event
			else if (sender == shapeButton) {
				ArrayList fixedNodes = graph.getFixed();
				ShapeDialog dialog = new ShapeDialog(lastClickedNode, fixedNodes);
				dialog.setLocationRelativeTo(this);

				dialog.setVisible(true);
				
			}
			else {
				//event fired by a button referencing a hyper link
				sender.setLinkVisited(true);
				appletRef.getAppletContext().showDocument(sender.getLinkURL(),
						"_blank");
			}

		} catch (Exception ex) {
			// do nothing
		}

		overviewRef.repaint();
		repaint();
	}

	/**
	 * 
	 *checks, if the grouping circles should be displayed further when switching between graph types(simple/exteded)
	 *and if the grouping button should be enabled or not
	 */
	public void checkShowGroups() {
		//confidence grouping has to be calculated again, if one switches between
		//simple/extended graph
		if (groupType.equalsIgnoreCase("Confidence") && showGroupBorders)
			showGroupBorders = false;
		//no grouping allowed, if nodes are collapsed
		if(collapsNodesCounter > 0){
			if(appletRef != null)
				appletRef.switchVisibilityGroupButton(false);
			showGroupBorders = false;
		}
		else{
			if(appletRef != null)
				appletRef.switchVisibilityGroupButton(true);
		}
			
	}

	/**
	 * 
	 * @param lowValue
	 * @param highValue
	 * 
	 * creates a list with those edges, which fulfil the confidence restrictions.
	 * only those edges will be displayed, which have a confidence between @param lowValue
	 * and @param highValue
	 */
	public void setEdgeRange(int lowValue, int highValue) {
		this.lowValue = lowValue;
		this.highValue = highValue;
//		int lowPos = 0;
//		int highPos = 0;

		//extended graph
		if (prettyEdge) {
//			lowPos = graph.findConfidencePos(lowValue / 10.F);
//			highPos = graph.findConfidencePos(highValue / 10.F);
			//subEdges = new ArrayList(graph.getEdges().subList(highPos, lowPos));
			
			subEdges = getSubList(graph.getEdges(), lowValue/10.F, highValue/10.F);
			
			//adding non-confidence edges
//			ArrayList nonConfList = ((JSquidGraph)graph).getNonConfEdges();
//			for (Iterator itor = nonConfList.iterator(); itor.hasNext();) {
//				Edge e = (Edge) itor.next();
//				if (!subEdges.contains(e))
//					subEdges.add(e);
//			}
		}
		//simple graph
		else {
//			lowPos = graph.findConfidencePosSingle(lowValue / 10.F);
//			highPos = graph.findConfidencePosSingle(highValue / 10.F);
			//subEdges = new ArrayList(graph.getSingleEdges().subList(highPos, lowPos));
		
			subEdges = getSubList(graph.getSingleEdges(), lowValue/10.F, highValue/10.F);
		}
		
		repaint();
	}
	
	private ArrayList getSubList(ArrayList originalList, float lowValue, float highValue){
		ArrayList subList = new ArrayList();
		
		for (Iterator itor = originalList.iterator(); itor.hasNext();) {
			Edge e = (Edge) itor.next();
			float conf = e.getConf();
			//if slider is at the highest level, edges with a higher confidence
			//than 1.0 should also be displayed
			if (highValue == 1.0f)
				highValue = Float.MAX_VALUE;
			if ((conf >= lowValue && conf <= highValue) || !e.isConfidenceEdge())
				subList.add(e);
		}
		
		return subList;
	}
	

	public void mouseMoved(MouseEvent e) {
		super.mouseMoved(e);
		Node node = getClosest(e);
		//highlight all edges connected with this node
		if (node != null) {
			areEdgesHighlighted = true;
			if (prettyEdge)
				graph.findAllEdges(node);
			else
				graph.findAllSingleEdges(node);
			repaint();
		} else if (areEdgesHighlighted) {
			if (prettyEdge)
				graph.disableEdgeHighlighting();
			else
				graph.disableSingleEdgeHighlighting();
			areEdgesHighlighted = false;
			repaint();
		}
	}
	
	public void mouseEntered(MouseEvent e) {
		Object sender = e.getSource();
		if (e.getSource() == nodePopup) 
			isPopupUsed = true;
		//not showing nodePopup any more, when mouse leaves popup area
		else if (sender == this && isPopupUsed) {
			nodePopup.setVisible(false);
			isPopupUsed = false;
		}
	}
	
//	public void mouseExited(MouseEvent e) {
//		if (e.getSource() ==this && !isPopupUsed)
//			nodePopup.setVisible(false);
//	}

}
