/**
 * JSquidPictureMaker.java
 */
package jsquid.picture;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStreamReader;

import jsquid.display.drawing.factory.DrawingType;
import jsquid.xml.InteractionClass;

import org.apache.commons.cli.*;

/**
 * @author sanjit
 * Picture Maker creates Pictures out of jSquid XML
 * uses commons-cli-1.1.jar
 */
public class JSquidPictureMaker {

	/**
	 * @param args
	 */
	public static void main(String[] args) {
		Options options = new Options();
		Exporter picmaker = null;
		createOptions(options);

		CommandLineParser parser = new GnuParser();
		CommandLine cmd = null;
		try {
			cmd = parser.parse(options, args);
		} catch (ParseException e) {
			e.printStackTrace();
		}

		try {
			picmaker = handleOptions(cmd, options);
		} catch (Exception e1) {
			showHelp(options);
		}
		if (picmaker != null) {
			try {
				picmaker.export();
			} catch (IOException e) {
				e.printStackTrace();
			}
		}

	}

	/**
	 * Handle options from console
	 * @param cmd
	 * @param options
	 * @return
	 * @throws Exception
	 */
	private static Exporter handleOptions(CommandLine cmd, Options options)
			throws Exception {
		String xmlInput = "";
		// String[] settings = new String[3];
		int width = 800;
		int height = 700;
		String filename = null;
		boolean showEdgeColor = true;
		boolean showEdgeConfidence = true;
		boolean showEdgeLabels = true;
		boolean showNodeSize = true;
		boolean showNodeLabels = true;
		boolean reqOption = false;
		String style = DrawingType.NORMALTYPE;
		String interactionType = InteractionClass.INTERACTIONCLASS_TYPES;

		if (cmd.hasOption("h")) {
			showHelp(options);
			return null;
		}
		//read from stdin
		if(cmd.hasOption("stdin")){
			BufferedReader stdin = new BufferedReader(new InputStreamReader(System.in));
			String line = "";
			while(((line = stdin.readLine()) != null) && (!line.equalsIgnoreCase("</jsquid>"))){
				xmlInput += line;
			}
			xmlInput += "</jsquid>";
			stdin.close();
			reqOption = true;
		}
		//read from console parameter
		if (cmd.hasOption("xml")) {
			xmlInput = cmd.getOptionValue("xml");
			reqOption = true;
		}
		//read from file
		if (cmd.hasOption("xmlfile")) {
			xmlInput = "";
			String xmlFilename = cmd.getOptionValue("xmlfile");
			BufferedReader reader = new BufferedReader(new FileReader(new File(
					xmlFilename)));
			String line = "";
			while ((line = reader.readLine()) != null)
				xmlInput += line + System.getProperty("line.separator");

			reqOption = true;
		}
		if (cmd.hasOption("width")) {
			width = Integer.parseInt(cmd.getOptionValue("width"));
		}
		if (cmd.hasOption("height")) {
			height = Integer.parseInt(cmd.getOptionValue("height"));
		}
		if (cmd.hasOption("file")) {
			filename = cmd.getOptionValue("file");
		}
		if (cmd.hasOption("edgecolor")) {
			if (cmd.getOptionValue("edgecolor").equalsIgnoreCase("off"))
				showEdgeColor = false;
		}
		if (cmd.hasOption("edgeconfidence")) {
			if (cmd.getOptionValue("edgeconfidence").equalsIgnoreCase("off"))
				showEdgeConfidence = false;
		}
		if (cmd.hasOption("edgelabel")) {
			if (cmd.getOptionValue("edgelabel").equalsIgnoreCase("off"))
				showEdgeLabels = false;
		}
		if (cmd.hasOption("nodesize")) {
			if (cmd.getOptionValue("nodesize").equalsIgnoreCase("off"))
				showNodeSize = false;
		}
		if (cmd.hasOption("nodelabel")) {
			if (cmd.getOptionValue("nodelabel").equalsIgnoreCase("off"))
				showNodeLabels = false;
		}
		if (cmd.hasOption("interaction")) {
			String interactionValue = cmd.getOptionValue("interaction");
			if (interactionValue.equalsIgnoreCase("species"))
				interactionType = InteractionClass.INTERACTIONCLASS_SPECIES;
			else if (interactionValue.equalsIgnoreCase("links"))
				interactionType = InteractionClass.INTERACTIONCLASS_LINKS;
		}
		if (cmd.hasOption("style")) {
			style = cmd.getOptionValue("style");
		}
		if (!reqOption) {
			showHelp(options);
			return null;
		}
		Exporter export = null;
		//Decide which format
		if (cmd.hasOption("format")) {
			if (cmd.getOptionValue("format").equalsIgnoreCase("eps")) {
				export = new ExportToEPS(xmlInput, width, height, filename,
						style, interactionType);
			}
		}
		if (export == null)
			export = new ExportToGIF(xmlInput, width, height, filename, style,
					interactionType);
		export.setPreferences(showEdgeColor, showEdgeConfidence,
				showEdgeLabels, showNodeSize, showNodeLabels);
		return export;

	}

	/**
	 * Shows help out of @param options
	 */
	private static void showHelp(Options options) {
		HelpFormatter formatter = new HelpFormatter();
		formatter.printHelp("PicMaker", options, true);
	}

	/**
	 * Create CommandLine Options
	 * @param options
	 */
	private static void createOptions(Options options) {
		options.addOption("h", false, "show help");
		options.addOption("stdin", false, "Read XML from Stdin");
		Option xml = OptionBuilder.withArgName("xml").hasArg().withDescription(
				"XML Text as Input").create("xml");
		Option xmlFile = OptionBuilder.withArgName("xmlfile").hasArg()
				.withDescription("XML File as Input").create("xmlfile");
		Option width = OptionBuilder.withArgName("width").hasArg()
				.withDescription(
						"specifiy the width of the picture. default: 800")
				.create("width");
		Option height = OptionBuilder.withArgName("height").hasArg()
				.withDescription(
						"specifiy the height of the picture. default: 700")
				.create("height");
		Option file = OptionBuilder.withArgName("file").hasArg()
				.withDescription(
						"Output file (optional) for example \"test.gif\"")
				.create("file");
		Option format = OptionBuilder.withArgName("format").hasArg()
				.withDescription("Which format: gif(default) or eps").create(
						"format");
		Option edgeColor = OptionBuilder
				.withArgName("edgecolor")
				.hasArg()
				.withDescription(
						"Turn on or off edge color (default: on). If off will turn off edge colors(Simple Graph).")
				.create("edgecolor");
		Option edgeConfidence = OptionBuilder.withArgName("edgeconfidence")
				.hasArg().withDescription(
						"Turn on or off edge confidence (default: on)").create(
						"edgeconfidence");
		Option edgeLabels = OptionBuilder.withArgName("edgelabel").hasArg()
				.withDescription("Turn on or off edge label (default: on)")
				.create("edgelabel");
		Option nodeSize = OptionBuilder.withArgName("nodesize").hasArg()
				.withDescription("Turn on or off node size (default: on)")
				.create("nodesize");
		Option nodeLabels = OptionBuilder.withArgName("nodelabel").hasArg()
				.withDescription("Turn on or off node label (default: on)")
				.create("nodelabel");
		Option typeOfInteraction = OptionBuilder
				.withArgName("interaction")
				.hasArg()
				.withDescription(
						"Which typ of interaction should be displayed (types, species, links) default: types")
				.create("interaction");
		Option typeOfStyle = OptionBuilder.withArgName("style").hasArg()
				.withDescription(
						"Draw Style (" + DrawingType.NORMALTYPE + ", "
								+ DrawingType.OLDTYPE + ") default: "
								+ DrawingType.NORMALTYPE).create("style");

		options.addOption(xml);
		options.addOption(xmlFile);
		options.addOption(width);
		options.addOption(height);
		options.addOption(file);
		options.addOption(format);
		options.addOption(edgeColor);
		options.addOption(edgeConfidence);
		options.addOption(edgeLabels);
		options.addOption(nodeSize);
		options.addOption(nodeLabels);
		options.addOption(typeOfInteraction);
		options.addOption(typeOfStyle);
	}
}
