/**
 * Exporter.java
 */
package jsquid.picture;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URL;

import jsquid.JSquidPanel;
import jsquid.xml.InteractionClass;
import jsquid.xml.XMLDataLoader;
import jsquid.xml.XMLFileNotValidException;

import medusa.MedusaSettings;
import medusa.display.FRspring;
import medusa.graph.Graph;

/**
 * @author sanjit
 * Abstract class Exporter for implementation of Picture Exporters
 */
public abstract class Exporter {
	private XMLDataLoader loader;

	private String xmlInput;

	protected int width;

	protected int height;

	protected String filename;

	private MedusaSettings typeSettings;
	
	private MedusaSettings speciesSettings;
	
	private MedusaSettings linkSettings;

	protected JSquidPanel panel;

	protected boolean toSysOut; //if write Output to System.out

	private String style; //style of graph

	private boolean showEdgeColor;

	private boolean showEdgeConfidence;

	private boolean showEdgeLabels;

	private boolean showNodeSize;

	private boolean showNodeLabels;

	private String interactionType;

	public Exporter(String xmlInput, int width, int height, String filename,
			String style, String interactionType) {
		this.xmlInput = xmlInput;
		this.width = width;
		this.height = height;
		this.filename = filename;
		this.style = style;
		this.interactionType = interactionType;
	}

	// This is a template function
	public final void export() throws IOException {
		toSysOut = false;

		InputStream xmlStream = new ByteArrayInputStream(xmlInput.getBytes());
		//load and validate file
		URL schemaURL = this.getClass().getResource(
				"/jsquid/files/jsquidschema.xsd");
		try {
			loader = new XMLDataLoader(schemaURL.toString(), xmlStream);
		} catch (XMLFileNotValidException e) {
			System.exit(1);
		}

		Graph graph = loader.loadParameters();
		typeSettings = new MedusaSettings(loader
				.getColorTable(InteractionClass.INTERACTIONCLASS_TYPES), loader
				.getInteractionTable(InteractionClass.INTERACTIONCLASS_TYPES),
				loader.getCheckedTable(InteractionClass.INTERACTIONCLASS_TYPES));
		speciesSettings = new MedusaSettings(loader
				.getColorTable(InteractionClass.INTERACTIONCLASS_SPECIES), loader
				.getInteractionTable(InteractionClass.INTERACTIONCLASS_SPECIES),
				loader.getCheckedTable(InteractionClass.INTERACTIONCLASS_SPECIES));
		linkSettings = new MedusaSettings(loader
				.getColorTable(InteractionClass.INTERACTIONCLASS_LINKS), loader
				.getInteractionTable(InteractionClass.INTERACTIONCLASS_LINKS),
				loader.getCheckedTable(InteractionClass.INTERACTIONCLASS_LINKS));
		
		int colorTableSizeSum = typeSettings.getSizeColorTable()
		+ speciesSettings.getSizeColorTable()
		+ linkSettings.getSizeColorTable();
		
//		medusaSettings = new MedusaSettings(loader
//				.getColorTable(interactionType), loader
//				.getInteractionTable(interactionType), loader.getCheckedTable());
		//init Panel
		panel = new JSquidPanel(typeSettings, null, null, colorTableSizeSum);
		panel.setSettings(typeSettings);
		panel.updateShow();
		switchSettings();
		
		//Do Fruchterman Reingold Force-Directed layout 
		FRspring frSpring = new FRspring(graph, width - 30, height - 50);
		frSpring.iterateAll();

		((JSquidPanel) panel).switchStyle(style);

		panel.setGraph(graph);

		//set layout preferences for panel
		setPanelPrefs();

		OutputStream stream = doExport();

		if (toSysOut)
			((ByteArrayOutputStream) stream).writeTo(System.out);

		stream.close();
	}

	/**
	 *	switch MedusaSettings(links or species)
	 */
	private void switchSettings() {
		if(interactionType.equals(InteractionClass.INTERACTIONCLASS_SPECIES)){
			panel.setSettings(speciesSettings);
			panel.updateShow();
		}
		else if(interactionType.equals(InteractionClass.INTERACTIONCLASS_LINKS)){
			panel.setSettings(linkSettings);
			panel.updateShow();
		}
		
	}

	/**
	 *	set Preferences of Panel (pretty, labels etc.).
	 */
	private void setPanelPrefs() {
		panel.prettyEdge = showEdgeColor;
		panel.showConfidence = showEdgeConfidence;
		panel.labelEdge = showEdgeLabels;
		panel.prettyNode = showNodeSize;
		panel.labelNode = showNodeLabels;
	}

	protected abstract OutputStream doExport() throws IOException;

	/**
	 * Set Preferences in class
	 * @param showEdgeColor
	 * @param showEdgeConfidence
	 * @param showEdgeLabels
	 * @param showNodeSize
	 * @param showNodeLabels
	 */
	public void setPreferences(boolean showEdgeColor,
			boolean showEdgeConfidence, boolean showEdgeLabels,
			boolean showNodeSize, boolean showNodeLabels) {
		this.showEdgeColor = showEdgeColor;
		this.showEdgeConfidence = showEdgeConfidence;
		this.showEdgeLabels = showEdgeLabels;
		this.showNodeSize = showNodeSize;
		this.showNodeLabels = showNodeLabels;
	}

}
