/**
 * XMLSAXWriter.java
 */
package jsquid.convert.writers;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URL;
import java.util.ArrayList;
import java.util.HashMap;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.Transformer;
import javax.xml.transform.sax.SAXTransformerFactory;
import javax.xml.transform.sax.TransformerHandler;
import javax.xml.transform.stream.StreamResult;

import medusa.MedusaSettings;

import org.xml.sax.SAXException;

/**
 * @author sanjit
 * This is a standard, portable, memory-friendly XML Writer based on JAXP 1.1 + SAX
 * Implementation of IXMLWriter
 */
public abstract class XMLSAXWriter implements IXMLWriter {

	protected TransformerHandler hd = null;

	private boolean toSysOut = false;

	private String filename;

	private HashMap nodes;

	private ArrayList edges;

	private MedusaSettings settings;

	private static final String schemaPath = "/jsquid/files/jsquidschema.xsd";

	public XMLSAXWriter(String filename, HashMap nodes, ArrayList edges,
			MedusaSettings settings) {
		this.filename = filename;
		this.nodes = nodes;
		this.edges = edges;
		this.settings = settings;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see jsquid.writers.IXMLWriter#writeXML()
	 */
	public void writeXML() throws Exception {
		SAXTransformerFactory tf = (SAXTransformerFactory) SAXTransformerFactory
				.newInstance();

		hd = tf.newTransformerHandler();

		Transformer serializer = hd.getTransformer();
		serializer.setOutputProperty(OutputKeys.ENCODING, "UTF-8");
		serializer.setOutputProperty(OutputKeys.METHOD, "xml");
		serializer.setOutputProperty(OutputKeys.INDENT, "yes");
		serializer.setOutputProperty("{http://xml.apache.org/xslt}indent-amount", "4");
		//initalize stream
		OutputStream stream = null;
		if (filename != null)
			stream = new FileOutputStream(filename);
		else {
			stream = new ByteArrayOutputStream();
			toSysOut = true;
		}

		StreamResult streamResult = new StreamResult(stream);
		hd.setResult(streamResult);

		try {
			hd.startDocument();

			doWriteXML();

			hd.endDocument();
			InputStream inStream = null;
			if (toSysOut) {
				((ByteArrayOutputStream) stream).writeTo(System.out);
				inStream = new ByteArrayInputStream(
						((ByteArrayOutputStream) stream).toByteArray());
			} else {
				inStream = new FileInputStream(filename);
			}

			stream.close();
			
			//Validate generated file
			URL schemaURL = this.getClass().getResource(
					schemaPath);
			validateFile(schemaURL.toString(), inStream);
			inStream.close();
		} catch (SAXException e) {
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		} catch (Exception e) {
			e.printStackTrace();
		}

	}

	/**
	 * validates a XML File
	 * @param schemaPath schema Location
	 * @param fileStream XML file which should be validated
	 * @throws Exception
	 */
	private void validateFile(String schemaPath, InputStream fileStream)
			throws Exception {
		boolean success = false;
		DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();

		factory.setValidating(true);

		factory.setAttribute(
				"http://java.sun.com/xml/jaxp/properties/schemaLanguage",
				"http://www.w3.org/2001/XMLSchema");
		factory.setAttribute(
				"http://java.sun.com/xml/jaxp/properties/schemaSource",
				schemaPath);
		try {
			DocumentBuilder parser = factory.newDocumentBuilder();
			parser.setErrorHandler(new XMLErrorHandler());
			// doc = parser.parse("jsquidexample.xml");
			parser.parse(fileStream);
			success = true;
		} catch (ParserConfigurationException e) {
			System.out.println(e.getMessage());
		} catch (SAXException e) {
			System.out.println(e.getMessage());
		} catch (IOException e) {
			System.out.println(e.getMessage());
		}
		if (!success) {
			System.out.println("xml file INVALID");
			throw new Exception("File not valid");
		}

	}

	protected abstract void doWriteXML() throws Exception;

	/**
	 * @return the edges
	 */
	protected final ArrayList getEdges() {
		return edges;
	}

	/**
	 * @return the nodes
	 */
	protected final HashMap getNodes() {
		return nodes;
	}

	/**
	 * @return the settings
	 */
	protected final MedusaSettings getSettings() {
		return settings;
	}

}
