/**
 * ConvertXML.java
 */
package jsquid.convert;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.InputStreamReader;

import jsquid.convert.readers.IXMLReader;
import jsquid.convert.readers.XGMMLDOMReader;
import jsquid.convert.writers.IXMLWriter;
import jsquid.convert.writers.JSquidSAXWriter;

import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.CommandLineParser;
import org.apache.commons.cli.GnuParser;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.OptionBuilder;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;

/**
 * @author sanjit
 * Converts XML(for Graph Viewers) from Format A to Format B
 */
public class ConvertXML {

	/**
	 * @param args
	 */
	public static void main(String[] args) {
		Options options = new Options();
		IXMLReader reader = null;
		IXMLWriter writer = null;

		createOptions(options);
		String filename = null;
		CommandLineParser parser = new GnuParser();
		CommandLine cmd = null;
		try {
			cmd = parser.parse(options, args);
		} catch (ParseException e) {
			e.printStackTrace();
		}

		try {
			reader = handleOptions(cmd, options);
		} catch (Exception e1) {
			showHelp(options);
		}

		if (reader != null) {
			try {
				reader.readXML();
				if (cmd.hasOption("outputfile")) {
					filename = cmd.getOptionValue("outputfile");
				}
				writer = new JSquidSAXWriter(filename, reader.getNodes(),
						reader.getEdges(), reader.getSettings());
				writer.writeXML();
			} catch (Exception e) {
				e.printStackTrace();
			}
		}

	}

	/**
	 * handle the commandline options
	 * @param cmd CommandLine
	 * @param options Options
	 * @return a new IXMLReader Object
	 * @throws Exception 
	 */
	private static IXMLReader handleOptions(CommandLine cmd, Options options) throws Exception {
		String xmlInput = "";
		boolean reqOption = false;
		final String endOfFileMarker = "</graph>";

		if (cmd.hasOption("h")) {
			showHelp(options);
			return null;
		}

		//		read from console parameter
		if (cmd.hasOption("xml")) {
			xmlInput = cmd.getOptionValue("xml");
			reqOption = true;
		}
		//		read from file
		if (cmd.hasOption("xmlfile")) {

			String xmlFilename = cmd.getOptionValue("xmlfile");
			BufferedReader reader = new BufferedReader(new FileReader(new File(
					xmlFilename)));
			String line = "";
			while ((line = reader.readLine()) != null)
				xmlInput += line + System.getProperty("line.separator");

			reqOption = true;
		}
		//		read from stdin
		if(cmd.hasOption("stdin")){
			BufferedReader stdin = new BufferedReader(new InputStreamReader(System.in));
			String line = "";
			while(((line = stdin.readLine()) != null) && (!line.equalsIgnoreCase(endOfFileMarker))){
				xmlInput += line;
			}
			xmlInput += endOfFileMarker;
			stdin.close();
			reqOption = true;
		}


		if (!reqOption) {
			showHelp(options);
			return null;
		}

		return new XGMMLDOMReader(xmlInput);
	}

	/**
	 * Shows the help of the program
	 * @param options
	 */
	private static void showHelp(Options options) {
		HelpFormatter formatter = new HelpFormatter();
		formatter.printHelp("ConvertXML", options, true);

	}

	/**
	 * Creates option for program
	 * @param options Options
	 */
	private static void createOptions(Options options) {
		options.addOption("h", false, "show help");
		options.addOption("stdin", false, "Read XML from Stdin");
		Option xml = OptionBuilder.withArgName("xml").hasArg().withDescription(
				"XML Text as Input").create("xml");
		Option xmlFile = OptionBuilder.withArgName("xmlfile").hasArg()
				.withDescription("XML File as Input").create("xmlfile");
		Option file = OptionBuilder
				.withArgName("outputfile")
				.hasArg()
				.withDescription(
						"Output file (optional) for example \"test.xml\" if not specified STDOUT will be used")
				.create("outputfile");

		options.addOption(xml);
		options.addOption(xmlFile);
		options.addOption(file);

	}

}
