// Medusa is a graph viewer which allows interactive editing of networks
// (edges and nodes) and also connects to databases.
//
// Copyright (C) 2006 Sean Hooper
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or (at
// your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License along
// with this program; if not, write to the
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA 02111-1307 USA

package medusa.graph;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.HashMap;
import java.util.Iterator;


/**
 * Class <code>Graph</code>: The Graph object is basically a collection of
 * edges with some utilities
 * 
 * @author <a href="mailto:hooper@embl.de">Sean Hooper</a>
 * @version 1.0
 */
public class Graph implements Serializable, Cloneable {
	// we will mostly add or remove to edges?
	protected ArrayList edges = new ArrayList();
	
	protected ArrayList singleEdges = new ArrayList();

	protected ArrayList uniqueEdges = new ArrayList();

	// nodes, on the other hand, need fast searching
	protected HashMap nodes = new HashMap();

	private HashMap groupTypes = new HashMap();

	// comment lines
	private String comment;

	private HashMap linkStubMap = null;

	private ArrayList listOfHits = new ArrayList();
	
	private HashMap connectionMap = null;
	

	public void setLinkStubMap(HashMap linkStubMap) {
		this.linkStubMap = linkStubMap;
	}

	public HashMap getLinkStubMap() {
		return this.linkStubMap;
	}

	public HashMap getGroupTypes() {
		return groupTypes;
	}

	public void setGroupTypes(HashMap groupTypes) {
		this.groupTypes = groupTypes;
	}

	public HashMap getGroup(String type) {
		return (HashMap) groupTypes.get(type);
	}

	public void setGroup(String type, HashMap group) {
		groupTypes.put(type, group);
	}
	
	public void setConnectionMap(HashMap connMap) {
		this.connectionMap = connMap;
		//initSingleEdges();
	}
	
	public HashMap getConnectionMap() {
		return this.connectionMap;
	}
	
	public ArrayList getSingleEdges() {
		return this.singleEdges;
	}

	public void sortEdgeList() {
		Collections.sort(edges);
		Collections.sort(singleEdges);
	}

	public int findConfidencePos(float conf) {
		
		return findConfidencePosition(conf, edges);
	}
	
	public int findConfidencePosSingle(float conf) {
		
		return findConfidencePosition(conf, singleEdges);
	}
	
	private int findConfidencePosition(float conf, ArrayList list) {
		if (conf == 1.0F)
			return 0;
		if (conf == 0.0F)
			return list.size();
		conf -= 0.0001F;
		int i = Collections.binarySearch(list, new Edge("", "", conf));
		if (i < 0) {
			++i;
			i = -i;
		}
		return i;
	}
	

	/**
	 * Add <CODE>Edge</CODE> to <CODE>Graph</CODE>
	 * 
	 * @param e
	 *            <CODE>Edge</CODE> to add
	 */
	public void addEdge(Edge e) {
		if ((!edges.contains(e)) & (e != null)) {
			// System.out.println("debug: adding edge "+e.report());
			edges.add(e);
			addNode(e.getFromName());
			addNode(e.getToName());
			// add unique edge, if it doesnt already exist
			UniqueEdge ue = new UniqueEdge(e);
			// System.out.println("debug: adding unique edge "+ue.report()+"
			// "+ue.getClass());
			if (!uniqueEdges.contains(ue))
				uniqueEdges.add(ue);
		}
	}
	
	public void addSingleEdge(Edge e) {
		if (e!=null && !singleEdges.contains(e)) {
			singleEdges.add(e);
		}
	}

	/**
	 * Adds an edge by creating it from two <CODE>Nodes</CODE>.
	 * 
	 * @param n1
	 * @param n2
	 * @param conf
	 * @param type
	 * @see <CODE>Node</CODE>
	 */
	public void addEdge(Node n1, Node n2, float conf, int type) {
		Edge e = new Edge(n1.getLabel(), n2.getLabel(), conf, type);
		if ((!edges.contains(e)) & (e != null)) {
			// System.out.println("debug: adding edge "+e.report());
			edges.add(e);
			addNode(n1);
			addNode(n2);
			// add unique edge, if it doesnt already exist
			UniqueEdge ue = new UniqueEdge(e);
			// System.out.println("debug: adding unique edge "+ue.report()+"
			// "+ue.getClass());
			if (!uniqueEdges.contains(ue))
				uniqueEdges.add(ue);
		}
	}

	public ArrayList getEdgeAL() {
		return edges;
	}

	public ArrayList getUniqueEdgeAL() {
		return uniqueEdges;
	}

	public HashMap getNodeHM() {
		return nodes;
	}

	public String getComment() {
		return comment;
	}

	/**
	 * Gets the value of edges
	 * 
	 * @return the value of edges
	 */
	public final ArrayList getEdges() {
		return this.edges;
	}

	/**
	 * Sets the value of edges
	 * 
	 * @param argEdges
	 *            Value to assign to this.edges
	 */
	public final void setEdges(final ArrayList argEdges) {
		this.edges = argEdges;
	}

	/**
	 * Gets the value of uniqueEdges
	 * 
	 * @return the value of uniqueEdges
	 */
	public final ArrayList getUniqueEdges() {
		return this.uniqueEdges;
	}

	/**
	 * Sets the value of uniqueEdges
	 * 
	 * @param argUniqueEdges
	 *            Value to assign to this.uniqueEdges
	 */
	public final void setUniqueEdges(final ArrayList argUniqueEdges) {
		this.uniqueEdges = argUniqueEdges;
	}

	/**
	 * Gets the value of nodes
	 * 
	 * @return the value of nodes
	 */
	public final HashMap getNodes() {
		return this.nodes;
	}

	/**
	 * Sets the value of nodes
	 * 
	 * @param argNodes
	 *            Value to assign to this.nodes
	 */
	public final void setNodes(final HashMap argNodes) {
		this.nodes = argNodes;
	}

	/**
	 * Sets the value of comment
	 * 
	 * @param argComment
	 *            Value to assign to this.comment
	 */
	public final void setComment(final String argComment) {
		this.comment = argComment;
	}

	public Object clone() throws CloneNotSupportedException {
		Graph g = (Graph) super.clone();
		g.edges = (ArrayList) this.edges.clone();
		g.uniqueEdges = (ArrayList) this.uniqueEdges.clone();
		g.nodes = (HashMap) this.nodes.clone();

		return g;
	}

	public Graph deepClone() {

		try {
			// ObjectOutputStream erzeugen
			ByteArrayOutputStream bufOutStream = new ByteArrayOutputStream();
			ObjectOutputStream outStream = new ObjectOutputStream(bufOutStream);

			// Objekt im byte-Array speichern
			outStream.writeObject(this);
			outStream.close();

			// Pufferinhalt abrufen
			byte[] buffer = bufOutStream.toByteArray();
			// ObjectInputStream erzeugen
			ByteArrayInputStream bufInStream = new ByteArrayInputStream(buffer);
			ObjectInputStream inStream = new ObjectInputStream(bufInStream);

			// Objekt wieder auslesen
			Object newObj = inStream.readObject();
			inStream.close();

			return (Graph) newObj;

		} catch (Exception ex) {
			ex.printStackTrace();
			return null;
		}

	}

	/*
	 * public Graph(Graph g){ edges=g.getEdgeAL();
	 * uniqueEdges=g.getUniqueEdges();
	 */

	// add an internal edge, i.e, add edge only if
	// both nodes are already connected
	/*
	 * public void addInternalEdge(Edge e){ if (edges.contains(e)) return;
	 * Object n1 = nodes.get(new Node(e.n1)); Object n2 = nodes.get(new
	 * Node(e.n2)); if ( (n1!=null) & (n2!=null) ) addEdge(e); }
	 */

	public void removeEdge(Edge e) {
		if (!edges.contains(e))
			return;
		edges.remove(e);
		// removeNode(e.getFromName());
		// removeNode(e.getToName());
		UniqueEdge ue = new UniqueEdge(e);
		if (!edges.contains(ue))
			uniqueEdges.remove(ue);
	}

	public void renameNodes(String oldName, String newName) {
		for (Iterator i = edgesIterator(); i.hasNext();) {
			Edge e = (Edge) i.next();
			e.renameNode(oldName, newName);
		}
		renameNode(oldName, newName);
	}

	public void addNode(String name) {
		Node n = new Node(name);
		n.addConnection();
		if (!nodes.containsKey(name)) {
			nodes.put(name, n);
			// System.out.println("added node "+n.getLabel());
		} else {
			n = getNode(name);
			// System.out.println("increased linkage in node "+n.getLabel());
			n.addConnection();
			nodes.put(name, n);

		}
	}

	public void addNode(Node n) {

		if (!nodes.containsKey(n.getLabel())) {
			// set connections to 1 and put in hashmap
			n.addConnection();

			nodes.put(n.getLabel(), n);
			// System.out.println("added node "+n.getLabel());
		} else {
			// if contains, add connection and set node to new settings
			Node n1 = getNode(n.getLabel());
			n1.addConnection();
			// n1.copyPos(n);
			nodes.put(n.getLabel(), n);
		}
	}

	public void setNode(Node n) {
		if (nodes.containsKey(n.getLabel())) {
			Node n2 = getNode(n.getLabel());
			/*
			 * System.out.println("color changed from "+n2.color+" to "+
			 * n.color);
			 */
			n2.setColor(n.getColor());
			n2.setShape(n.getShape());
			// System.out.println("Setting "+n2.getLabel()+" to "+n.getX()
			// +" "+n.getY());
			n2.setX(n.getX());
			n2.setY(n.getY());
			n2.setAnnotation(n.getAnnotation());
		}
	}

	/*
	 * Copy the color settings (and other stuff eventually) from another graph
	 */
	public void copyNodeSettings(Graph temp) {
		if (temp == null)
			return;
		for (Iterator i = temp.nodesIterator(); i.hasNext();) {
			Node n = (Node) i.next();
			setNode(n);
		}
	}

	public void clear() {
		edges.clear();
		nodes.clear();
		uniqueEdges.clear();
		comment = "";
	}

//	private void removeNode(String name) {
//		// System.out.println("removing node "+name);
//		Node n = getNode(name);
//		nodes.remove(name);
//		if (n.removeConnection()) {
//			// System.out.println("replacing node "+n.name);
//			nodes.put(name, n);
//		}
//	}

	// private void removeNode(Node n) {
	// removeNode(n.getLabel());
	// }

	public Node getNode(String name) {
		return (Node) nodes.get(name);
	}

	public void renameNode(String oldName, String newName) {
		Node n = (Node) nodes.remove(oldName);
		if (n != null) {
			n.setLabel(newName);
			nodes.put(newName, n);
		}

	}

	public Iterator nodesIterator() {
		return nodes.values().iterator();
	}

	public Iterator edgesIterator() {
		return edges.iterator();
	}

	public Iterator edgesIterator(String lbl) {
		ArrayList temp = new ArrayList();
		for (Iterator i = edgesIterator(); i.hasNext();) {
			Edge e = (Edge) i.next();
			if (e.contains(lbl))
				temp.add(e);
		}
		return temp.iterator();
	}

	public Iterator uniqueEdgesIterator() {
		return uniqueEdges.iterator();
	}

	public ArrayList getCollapsed() {
		ArrayList collapsedNodes = new ArrayList();
		for (Iterator i = nodesIterator(); i.hasNext();) {
			Node n = (Node) i.next();
			if (n.isCollapseNode())
				collapsedNodes.add(n);
		}
		return collapsedNodes;
	}

	/**
	 * Returns an ArrayList of the fixed nodes
	 * 
	 * @return
	 */
	public ArrayList getFixed() {
		ArrayList fixedNodes = new ArrayList();
		for (Iterator i = nodesIterator(); i.hasNext();) {
			Node n = (Node) i.next();
			// edges.add(e);
			if (n.isFixed()) {
				fixedNodes.add(n);
				// System.out.println("added "+n.getLabel()+" to fixed list");
			}
		}
		return fixedNodes;
	}

	/*
	 * public void removeFixed(){ ArrayList<Node> a = getFixed(); for (Iterator
	 * i = a.iterator();i.hasNext();){ Node n = (Node) i.next();
	 * 
	 * removeNode(n); System.out.println("removed "+n.getLabel()+" from nodes"); } }
	 */
	public void removeFixed() {
		ArrayList a = getFixed();
		List list = (List) edges.clone();
		for (Iterator i = list.iterator(); i.hasNext();) {
			Edge e = (Edge) i.next();
			String[] names = e.getNodes();
			if (a.contains(new Node(names[0]))
					|| a.contains(new Node(names[1]))) {
				removeEdge(e);
				// System.out.println("removed "+e+" from edges");
			}
		}
	}

	public Node[] nodeArray() {
		Collection c = nodes.values();
		return (Node[]) c.toArray(new Node[0]);
	}

	public void rescaleNodes(int scale) {
		for (Iterator i = nodesIterator(); i.hasNext();) {
			Node n = (Node) i.next();
			n.rescale(scale);
		}
	}

	public void rescaleNodes(int xScale, int yScale) {
		for (Iterator i = nodesIterator(); i.hasNext();) {
			Node n = (Node) i.next();
			n.rescale(xScale, yScale);
		}
	}

	public HashMap nodesMap() {
		// we need indexes for nodes for the adjacency matrix
		HashMap hm = new HashMap();
		int count = 0;
		for (Iterator i = nodesIterator(); i.hasNext(); count++) {
			Node n = (Node) i.next();
			hm.put(n.getLabel(), new Integer(count));

		}
		return hm;
	}

	/**
	 * Divides node positions by a double value Useful for batch operations
	 * after using FRSpring
	 */
	public void divideNodePosition(double d) {
		for (Iterator i = nodesIterator(); i.hasNext();) {
			Node n = (Node) i.next();
			n.setX(n.getX() / d);
			n.setY(n.getY() / d);

		}

	}

	public Edge[] edgeArray() {
		return (Edge[]) edges.toArray(new Edge[0]);
	}

	/*
	 * public UniqueEdge[] uniqueEdgeArray(){ return (UniqueEdge[])
	 * uniqueEdges.toArray(new UniqueEdge[0]); }
	 */

	public void addGraph(Graph g) {

		for (Iterator i = g.edgesIterator(); i.hasNext();) {
			Edge e = (Edge) i.next();
			// edges.add(e);
			addEdge(e);
		}
		/*
		 * for (Iterator i = g.nodesIterator();i.hasNext();){ Node n = (Node)
		 * i.next();
		 * 
		 * addNode(n.lbl); }
		 */
	}

	/*
	 * public Graph clone(){ Graph g = new Graph(); g=this; return g; }
	 */

	public void subtractGraph(Graph g) {
		for (Iterator i = g.edgesIterator(); i.hasNext();) {
			Edge e = (Edge) i.next();

			removeEdge(e);
		}
	}

	public int getNodeSize() {
		return nodes.size();
	}

	public int getEdgeSize() {
		return edges.size();
	}

	public int getUniqueEdgeSize() {
		return uniqueEdges.size();
	}

	// public void balanceGraph(){
	// int maxIndex=0;
	// int maxCon=0;
	// int count=0;
	// // find the most connected node
	// for(Iterator i = nodes.values().iterator();i.hasNext();count++){
	// Node n = (Node) i.next();
	// if (n.getConnections() > maxCon){
	// maxIndex=count;
	//
	// //representative=n.name;
	// maxCon = n.getConnections();
	// }
	// }
	// //representative = maxIndex;
	// }

	public void cropEdges(double confidenceCutoff) {
		// cut out all edges which fail to reach the cutoff
		// copy array - never modify the structure you are iterating over
		List list = (List) edges.clone();
		for (Iterator i = list.iterator(); i.hasNext();) {
			Edge temp = (Edge) i.next();
			if (temp.getConf() < confidenceCutoff)
				removeEdge(temp);
		}
	}

	public void removeEdgeByLabel(String lbl) {
		List list = (List) edges.clone();
		for (Iterator i = list.iterator(); i.hasNext();) {
			Edge temp = (Edge) i.next();
			if (temp.contains(lbl)) {
				// System.out.println("found "+lbl);
				removeEdge(temp);
			}
		}
	}

	// public void renameNode(String from, String to){
	// Node n = getNode(from);

	// public void tracePath(String nodeName){
	// // put unresolved edges in a new set
	// ArrayList unresolved = new ArrayList();
	// unresolved = (ArrayList)edges.clone();
	// Graph resolved = new Graph();
	// Node n1=new Node(nodeName);
	// // start
	// Edge e;
	// while ((e=getAdjacent(unresolved,n1))!=null ){
	// unresolved.remove(e);
	// Node n2 = e.getComplement(n1.getLabel());
	// tracePath(unresolved,n2);
	//
	// System.out.print(n2.getLabel()+"-");
	// System.out.println(n1.getLabel());
	// }
	// }

	public void tracePath(ArrayList unresolved, Node n) {
		Edge e;
		while ((e = getAdjacent(unresolved, n)) != null) {
			unresolved.remove(e);
			Node n2 = e.getComplement(n.getLabel());
			tracePath(unresolved, n2);
			System.out.print(n2.getLabel() + "-");
		}
	}

	// get the next edge that contains the node
	public Edge getAdjacent(ArrayList unresolved, Node node) {
		for (Iterator i = unresolved.iterator(); i.hasNext();) {
			Edge e = (Edge) i.next();
			Node n = e.getComplement(node.getLabel());
			if (n != null)
				return e;
		}
		return null;
	}

	public String reportUnique() {
		StringBuffer sb = new StringBuffer();
		for (Iterator i = uniqueEdges.iterator(); i.hasNext();) {
			UniqueEdge e = (UniqueEdge) i.next();
			sb.append("\n");
			sb.append(e.getFromName());
			sb.append("\t");
			sb.append(e.getToName());
		}
		return sb.toString();
	}

	public String report() {
		StringBuffer sb = new StringBuffer();
		if (comment != null) {
			sb.append(comment);
			sb.append("\n");
		}

		sb.append("*edges");
		for (Iterator i = edges.iterator(); i.hasNext();) {
			Edge e = (Edge) i.next();
			sb.append("\n");
			sb.append(e.report());

		}

		sb.append("\n*nodes\n");
		for (Iterator i = nodes.values().iterator(); i.hasNext();) {
			Node n = (Node) i.next();
			sb.append(n.report());
			sb.append("\n");
		}

		return sb.toString();
	}

	public void addComment(String line) {
		if (line == null)
			return;
		if (line.length() > 1)
			comment = line;
	}

	public void defaultGraph() {
		Edge e1 = new Edge("n1", "n2", 4);
		Edge e2 = new Edge("n2", "n3", 5);
		Edge e3 = new Edge("n1", "n2", 5);
		Edge e4 = new Edge("n1", "n4", 1);
		Edge e5 = new Edge("n2", "n3", 1);
		addEdge(e1);
		addEdge(e2);
		addEdge(e3);
		addEdge(e4);
		addEdge(e5);
	}

	/*
	 * // scans for multiple edges and fixes orientation public void
	 * fixOrientation(Edge e1){
	 * 
	 * for(Iterator i = edges.iterator();i.hasNext();){ Edge e2 =
	 * (Edge)i.next(); if (e1.sameName(e2)==true) System.out.println(e2); } }
	 */

	public void setNodeColor(java.awt.Color color) {
		for (Iterator i = nodes.values().iterator(); i.hasNext();) {
			Node n = (Node) i.next();
			n.setColor(color);

		}
	}

	public void setNodeShape(int shape) {
		for (Iterator i = nodes.values().iterator(); i.hasNext();) {
			Node n = (Node) i.next();
			n.setShape(shape);

		}
	}

	public void setNodeAnnotation(StringBuffer annotation) {
		String[] result = annotation.toString().split("\n");
		String[] parts;
		Node n;
		for (int i = 0; i < result.length; i++) {
			parts = result[i].split("\t");
			n = (Node) nodes.get(parts[0]);
			if (n != null) {
				n.setAnnotation(parts[1]);
				// System.out.println(n.getLabel()+" set to
				// "+n.getAnnotation());
			}
		}
	}

	public void setNodeAnnotation(String name, String annotation) {
		Node n = (Node) nodes.get(name);
		if (n != null) {
			n.setAnnotation(annotation);
			// System.out.println(n.getLabel()+" set to "+n.getAnnotation());
		}
	}

	public void printNodeAnnotation() {
		for (Iterator i = nodes.values().iterator(); i.hasNext();) {
			Node n = (Node) i.next();
			System.out.println(n.getAnnotation());
		}
	}

	// return string list of nodes
	public StringBuffer getNodeList() {
		StringBuffer sb = new StringBuffer();
		for (Iterator i = nodes.values().iterator(); i.hasNext();) {
			Node n = (Node) i.next();
			// sb.append("node:");
			sb.append(n.getLabel());
			sb.append("\n");
		}
		return sb;
	}

	// get the subgraph that contains the node
	public Graph subGraph(Node node) {
		Graph sub = new Graph();
		Edge e;
		Node n;

		for (Iterator i = edgesIterator(); i.hasNext();) {
			e = (Edge) i.next();
			if (e.containsNode(node)) {
				sub.addEdge(e);
				n = e.getComplement(node);
				setNode(n);
			}
		}
		return sub;
	}

	public void rescaleConfidence() {
		Edge e;
		float max = 0.0f;
		for (Iterator i = edgesIterator(); i.hasNext();) {
			e = (Edge) i.next();
			if (e.getConf() > max)
				max = e.getConf();
		}
		for (Iterator i = edgesIterator(); i.hasNext();) {
			e = (Edge) i.next();
			e.setConf(e.getConf() / max);

		}
	}

	/*
	 * // get number of edges from this node // since this is a sparse graph,
	 * this // is not very efficient. In an adjacency // matrix, it would be
	 * much faster public int numberOfEdges(Node node){ UniqueEdge e; Node n;
	 * int neighbours=0; for(Iterator i = edgesIterator();i.hasNext();){ e =
	 * (UniqueEdge) i.next(); if (e.containsNode(node)){ neighbours++; } }
	 * return neighbours; }
	 */

	final int MAX_PARALLEL = 20;

	// improved but slower orientation algorithm
	public void autoFixOrientation() {

		Edge[] edgeArray = edgeArray();
		int nedges = edgeArray.length;
		// a list of edges already sorted out
		boolean[] edgeDone = new boolean[nedges];
		for (int i = 0; i < nedges; i++)
			edgeDone[i] = false;

		// location of multiple edges
		int[] edgeList = new int[MAX_PARALLEL];
		for (int i = 0; i < MAX_PARALLEL; i++)
			edgeList[i] = 0;
		double alpha = 1.; // the bezier attack point
		double startAlpha; // initial alpha point
		int matches = 0;
		// go through the edges and look for parallel edges
		for (int i = 0; i < nedges; i++) {
			// skip if edge already done
			// System.out.println("Checking edge "+i+" : "+edges[i]);
			if (edgeDone[i]) {
				// System.out.println("edge "+i+" done: "+edges[i]);
				continue;
			}
			matches = 0;
			edgeList[0] = i;
			for (int j = 0; j < nedges; j++) {
				if (i == j)
					continue;
				// check for a match
				if (edgeArray[i].sameName(edgeArray[j])) {
					// put the edge number in the list
					// System.out.println("edge "+i+" matches "+j);
					// System.out.println(edges[i]+" = "+edges[j]);
					matches++;
					edgeList[matches] = j;
					edgeDone[j] = true;
				}
			}
			// no match? then do nothing
			if (matches == 0) {
				edgeArray[i].setOrientation(0.);
				continue;
			}
			// match=1 means 2 edges... adjust for this

			// start with the case where there is an even number of
			// edges. NOTE that 3 matches = 4 edges!
			// System.out.println(" Found "+matches+" matches");
			if (matches % 2 == 1)
				startAlpha = matches * -alpha / 2;

			else
				// uneven number of edges
				startAlpha = matches * -alpha / 2;

			// System.out.println("Setting start alpha to "+startAlpha);
			for (int k = 0; k <= matches; k++) {
				edgeArray[edgeList[k]].setOrientation(isOppositeEdge(
						edgeArray[edgeList[k]], edgeArray[i])
						* (startAlpha + k * alpha));
				// System.out.println("orientation set to
				// "+edges[edgeList[k]].orientation);
			}
		}

	}

	// return a modifier of -1 if edges are opposite
	public double isOppositeEdge(Edge e1, Edge e2) {
		if (e1.oppositeName(e2))
			return -1.;
		return 1.;
	}

	// adjust the node x and y according to the new panel sizes
	public void rescaleNodes(double scale) {
		// double xr=(double) x / (double) defaultSize;
		// double yr=(double) y / (double) defaultSize;
		// double reScale=scale/oldScale;
		for (Iterator it = nodesIterator(); it.hasNext();) {
			Node node = (Node) it.next();
			// node.setX(node.getX() * xr);
			// node.setY(node.getY() * yr);
			node.setX(node.getX() * scale);
			node.setY(node.getY() * scale);
		}
	}

	public void setHitNodes(ArrayList listHits) {
		if (this.listOfHits != null) {
			for (Iterator iterOldHits = this.listOfHits.iterator(); iterOldHits
					.hasNext();) {
				String name = (String) iterOldHits.next();
				if (nodes.get(name) != null)
					((Node) nodes.get(name)).setSearchHit(false);
			}
		}
		if (listHits != null) {
			for (Iterator iterHits = listHits.iterator(); iterHits.hasNext();) {
				String name = (String) iterHits.next();
				if (nodes.get(name) != null)
					((Node) nodes.get(name)).setSearchHit(true);
			}
			this.listOfHits = new ArrayList(listHits);
		} else
			this.listOfHits = listHits;
	}

	public void findAllEdges(Node node) {
		for (Iterator itor = edgesIterator(); itor.hasNext();) {
			Edge e = (Edge) itor.next();
			if (e.contains(node.getLabel()))
				e.setHighlightedEdge(true);
		}
	}
	
	public void findAllSingleEdges(Node node) {
		for (Iterator itor = singleEdges.iterator(); itor.hasNext();) {
			Edge e = (Edge) itor.next();
			if (e.contains(node.getLabel()))
				e.setHighlightedEdge(true);
		}
	}

	public void disableEdgeHighlighting() {
		for (Iterator itor = edgesIterator(); itor.hasNext();)
			((Edge) itor.next()).setHighlightedEdge(false);
	}
	
	public void disableSingleEdgeHighlighting() {
		for (Iterator itor = singleEdges.iterator(); itor.hasNext();)
			((Edge) itor.next()).setHighlightedEdge(false);
	}
	

	
}
