// Medusa is a graph viewer which allows interactive editing of networks
// (edges and nodes) and also connects to databases.
//
// Copyright (C) 2006 Sean Hooper
//
// This program is free software; you can redistribute it and/or modify 
// it under the terms of the GNU General Public License as published by 
// the Free Software Foundation; either version 2 of the License, or (at 
// your option) any later version.
//
// This program is distributed in the hope that it will be useful, but 
// WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General 
// Public License for more details.
//
// You should have received a copy of the GNU General Public License along 
// with this program; if not, write to the 
// Free Software Foundation, Inc., 
// 59 Temple Place, Suite 330, 
// Boston, MA 02111-1307 USA

/*
 * Edge.java
 *
 * Created on den 9 februari 2005, 23:18
 */

package medusa.graph;


/**
 * <CODE>Edge</CODE> object extended from <CODE>BasicEdge</CODE>
 * 
 * @author Sean Hooper, Uno Torstensson
 */
public class Edge extends BasicEdge implements Comparable {
	private float conf;

	private double orientation = 0.;

	private int type;
	
	private boolean isConfidenceEdge = true;
	
	private boolean isHighlighted = false;

	/**
	 * Constructor
	 * 
	 * @param n1
	 *            Node 1
	 * @param n2
	 *            Node 2
	 * @param conf
	 *            Confidence level
	 * @param type
	 *            <CODE>int</CODE> type
	 */
	public Edge(String n1, String n2, float conf, int type) {
		this.n1 = n1;
		this.n2 = n2;

		this.conf = conf;
		this.type = type;
	}

	/**
	 * Constructor
	 * 
	 * @param n1
	 *            Node 1
	 * @param n2
	 *            Node 2
	 * @param conf
	 *            <CODE>float</CODE> Confidence level
	 * @param type
	 *            <CODE>int</CODE> type
	 * @param orientation
	 *            <CODE>double</CODE> orientation
	 */
	public Edge(String n1, String n2, float conf, int type, double orientation) {
		this.n1 = n1;
		this.n2 = n2;

		this.conf = conf;
		this.type = type;
		this.orientation = orientation;
	}

	public Edge(String n1, String n2, Integer type, Double orientation,
			Float conf) {
		// System.out.println("constructor A");
		this.n1 = n1;
		this.n2 = n2;

		this.conf = conf.floatValue();
		this.type = type.intValue();
		this.orientation = orientation.doubleValue();

	}

	public Edge(String n1, String n2, Integer type, Float conf,
			Double orientation) {

		this(n1, n2, type, orientation, conf);
		// System.out.print("relayed to ");

	}

	/**
	 * Constructor
	 * 
	 * @param n1
	 *            Node 1
	 * @param n2
	 *            Node 2
	 * @param type
	 *            <CODE>int</CODE> type
	 * @param conf
	 *            Confidence level
	 * @param orientation
	 *            <CODE>double</CODE> orientation
	 */
	public Edge(String n1, String n2, int type, float conf, double orientation) {
		this(n1, n2, conf, type, orientation);
	}

	/**
	 * Constructor
	 * 
	 * @param n1
	 *            Node 1
	 * @param n2
	 *            Node 2
	 */
	public Edge(String n1, String n2) {
		this(n1, n2, 1.0f, 1);
	}

	/**
	 * Constructor
	 * 
	 * @param n1
	 *            Node 1
	 * @param n2
	 *            Node 2
	 * @param type
	 *            <CODE>int</CODE> type
	 */
	public Edge(String n1, String n2, int type) {
		this(n1, n2, 1.0f, type);
	}

	/**
	 * Constructor
	 * 
	 * @param n1
	 *            Node 1
	 * @param n2
	 *            Node 2
	 * @param conf
	 *            Confidence level
	 */
	public Edge(String n1, String n2, float conf) {
		this(n1, n2, conf, 1);
	}

	/**
	 * Set interaction type. Edges with different types of interactions will be
	 * displayed as multiple edges
	 * 
	 * @param type
	 *            <CODE>int</CODE> type
	 */
	public void setType(int type) {
		this.type = type;
	}

	/**
	 * Get the interaction type
	 * 
	 * @return <CODE>int</CODE> type
	 */
	public int getType() {
		return type;
	}
	
	public void setHighlightedEdge(boolean isHighlighted) {
		this.isHighlighted = isHighlighted;
	}
	
	public boolean isHighlightedEdge() {
		return this.isHighlighted;
	}
	

	/**
	 * Set confidence level
	 * 
	 * @param conf
	 *            <CODE>float</CODE> confidence 0.0 to 1.0
	 */
	public void setConf(float conf) {
		this.conf = conf;
	}

	/**
	 * Gets the confidence level
	 * 
	 * @return <CODE>float</CODE> confidence 0.0 to 1.0
	 */
	public float getConf() {
		return conf;
	}

	// if an edge contains a node, return the other node
	/**
	 * if an edge contains a node, return the other node
	 * 
	 * @param name
	 *            <CODE>String</CODE> name of node
	 * @return <CODE>String</CODE> name of other node
	 */
	public Node getComplement(String name) {
		if (n1.compareTo(name) == 0)
			return new Node(n2);
		if (n2.compareTo(name) == 0)
			return new Node(n1);
		return null;
	}

	/**
	 * Returns the complementary node
	 * 
	 * @param n
	 *            <CODE>Node</CODE> first node
	 * @return <CODE>Node</CODE> other node
	 */
	public Node getComplement(Node n) {
		return getComplement(n.getLabel());
	}

	public void setComplement(String n, String name){
		if(n1.compareTo(n) == 0)
			n2 = name;
		else if(n2.compareTo(n) == 0)
			n1 = name;
	}
	
	/**
	 * Compares two edges
	 * 
	 * @param o
	 *            other edge
	 * @return <CODE>True</CODE> if edges are the same
	 */
	public boolean equals(Object o) {
		// System.out.println("equals running..."+o.getClass());
		if (o.getClass() == this.getClass()) {
			// System.out.println("testing edge");
			Edge e = (Edge) o;
			return sameName(o) && (type == e.type) && (orientation == e.orientation);
		}

		// System.out.println("testing uniqueEdge");
		return sameName(o);
	}

	public String toString() {
		String result = n1 + "---" + n2;
		return result;
	}

	/**
	 * 
	 * @return
	 */
	public String report() {
		return super.report() + "\ti " + type + "\tc " + conf + "\to "
				+ orientation;
	}

	// public Node getNode1(){

	/**
	 * 
	 * @see BasicEdge
	 * @param n
	 * @return
	 */
	public boolean containsNode(Node n) {
		String name = n.getLabel();
		if (name.compareTo(n1) == 0)
			return true;
		if (name.compareTo(n2) == 0)
			return true;
		return false;
	}

	public void setOrientation(double orientation) {
		this.orientation = orientation;
	}

	/**
	 * Gets the current orientation
	 * 
	 * @return <CODE>double</CODE> orientation
	 */
	public double getOrientation() {
		return orientation;
	}

	public void renameNode(String oldName, String newName) {
		if (n1.compareTo(oldName) == 0) {
			n1 = newName;
		}
		if (n2.compareTo(oldName) == 0) {
			n2 = newName;
		}
	}

	public int compareTo(Object arg) {
		float anotherConfidence = ((Edge) arg).getConf();
		float compare = this.conf - anotherConfidence;
		if(compare == 0) return 0;
		else if(compare < 0) return 1;
		else return -1;
	}
	
	public Object clone(){
		Edge e = new Edge(this.n1, this.n2, this.type, this.conf, this.orientation);
		e.setConfidenceEdge(this.isConfidenceEdge);
		return e;
	}

	public void switchEdges() {
		String help = n2;
		n2 = n1;
		n1 = help;
	}

	public void setConfidenceEdge(boolean isConfidenceEdge) {
		this.isConfidenceEdge = isConfidenceEdge;
	}

	public boolean isConfidenceEdge() {
		return isConfidenceEdge;
	}

}
