// Medusa is a graph viewer which allows interactive editing of networks
// (edges and nodes) and also connects to databases.
//
// Copyright (C) 2006 Sean Hooper
//
// This program is free software; you can redistribute it and/or modify 
// it under the terms of the GNU General Public License as published by 
// the Free Software Foundation; either version 2 of the License, or (at 
// your option) any later version.
//
// This program is distributed in the hope that it will be useful, but 
// WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General 
// Public License for more details.
//
// You should have received a copy of the GNU General Public License along 
// with this program; if not, write to the 
// Free Software Foundation, Inc., 
// 59 Temple Place, Suite 330, 
// Boston, MA 02111-1307 USA

package medusa.display;

import java.awt.*;
import java.awt.geom.*;
import java.text.DecimalFormat;

import medusa.graph.Node;

/**
 * Convenience class containing methods of painting nodes and edges. Medusa will
 * most often use these methods. If you want to implement your own methods,
 * simply modify the methods <CODE>paintNode</CODE> and <CODE>paintEdge</CODE>
 * in the graph panels.
 */
public class PaintTools {

	public static final int SHAPE_COUNT = 10;
	
	public static final int SUBCELL_SHAPESIZE_CONST = 50;
	
	private static void drawArrow(Graphics2D g, int x1, int y1, int x2, int y2,
			double orientation, double offset) {
		double theta2;
		int lengthdeltaX;
		int lengthdeltaY;
		int widthdeltaX;
		int widthdeltaY;
		int headLength = 9;
		int headWidth = 6;

		theta2 = getTheta(x1, x2, y1, y2);

		// theta+=orientation*offset;
		theta2 -= orientation * offset;
		lengthdeltaX = -(int) (Math.cos(theta2) * headLength);
		lengthdeltaY = -(int) (Math.sin(theta2) * headLength);
		widthdeltaX = (int) (Math.sin(theta2) * headWidth);
		widthdeltaY = (int) (Math.cos(theta2) * headWidth);
		x2 -= (int) (Math.cos(theta2) * headLength);
		y2 -= (int) (Math.sin(theta2) * headLength);
		// g.drawLine(x1,y1,x2,y2);
		// g.drawLine(x2,y2,x2+lengthdeltaX+widthdeltaX,y2+lengthdeltaY-widthdeltaY);
		// g.drawLine(x2,y2,x2+lengthdeltaX-widthdeltaX,y2+lengthdeltaY+widthdeltaY);
		int[] xpoints = { x2 + lengthdeltaX + widthdeltaX, x2,
				x2 + lengthdeltaX - widthdeltaX };
		int[] ypoints = { y2 + lengthdeltaY - widthdeltaY, y2,
				y2 + lengthdeltaY + widthdeltaY };
		int npoints = 3;
		g.fillPolygon(xpoints, ypoints, npoints);

	}

	/**
	 * The theta parameter is used when drawing bezier curves.
	 * 
	 * @param x1
	 * @param x2
	 * @param y1
	 * @param y2
	 * @return theta: the "attack" parameter of bezier curves
	 */
	public static double getTheta(int x1, int x2, int y1, int y2) {
		int deltaX = (x2 - x1);
		int deltaY = (y2 - y1);
		double theta;
		theta = Math.atan((double) (deltaY) / (double) (deltaX));

		if (deltaX < 0.0) {
			theta = theta += Math.PI;
		}
		return theta;
	}

	
	
	public static Polygon rhomb(int x, int y, int nodeSize) {
		// Polygon rhombP = new Polygon();
		int offset = 2;
		int x1 = (int) (x + nodeSize / 2);
		int x2 = (int) (x + nodeSize + offset);
		int y1 = (int) (y + nodeSize / 2);
		int y2 = (int) (y + nodeSize);
		int[] xpoints = { x - offset, x1, x2, x1 };
		int[] ypoints = { y1, y2, y1, y };
		int npoints = 4;
		return new Polygon(xpoints, ypoints, npoints);
	}

	public static Polygon triangle(int x, int y, int nodeSize) {
		int x1 = (int) (x + nodeSize / 2);
		int x2 = (int) (x + nodeSize);
		int y2 = (int) (y + nodeSize);
		int[] xpoints = { x, x2, x1 };
		int[] ypoints = { y2, y2, y };
		int npoints = 3;
		return new Polygon(xpoints, ypoints, npoints);
	}

	public static Shape getShape(int i, int x, int y, int nodeSize) {
		switch (i) {

		case 0:
			return new Ellipse2D.Double(x, y, nodeSize, nodeSize);
		case 1:
			return new Rectangle2D.Double(x, y, nodeSize, nodeSize);
		case 2:
			return PaintTools.triangle(x, y, nodeSize);
		case 3:
			return PaintTools.rhomb(x, y, nodeSize);
		case 4:
			return PaintTools.leftTriangle(x, y, nodeSize);
		case 5:
			return PaintTools.rightTriangle(x, y, nodeSize);
		case 6:
			return PaintTools.revTriangle(x, y, nodeSize);
		case 7:
			return PaintTools.cross(x, y, nodeSize);
//		case 8:
//			return PaintTools.diagonalCross(x, y, nodeSize);
		case 8:
			return PaintTools.hexagon(x, y, nodeSize);
		case 9:
			return PaintTools.star(x, y, nodeSize);
		default:
			return new Ellipse2D.Double(x, y, nodeSize, nodeSize);
		}
	}
	
	/**
	 * A composite node has two or more colors. Nodes have additional data
	 * storage for additional colors. A composite node will be split into two
	 * halves with each half in a different color.
	 * 
	 * @param g2d
	 * @param n
	 * @param nodeSize
	 */
	public static void drawCompositeNode(Graphics2D g2d, final Node n,
			final int nodeSize) {
		int corr = (int) (nodeSize / 2.);
		int x = (int) n.getX();
		int y = (int) n.getY();
		Color c2 = n.getColor2();
		Color c3 = n.getColor3();
		Rectangle2D.Double nodeR = new Rectangle2D.Double(x - corr, y - corr,
				nodeSize, nodeSize);
		// draw basic shape
		g2d.setColor(n.getColor());
		g2d.fill(nodeR);
		// draw remaining colors
		if (c2 != null) {
			g2d.setColor(c2);
			g2d.fill(new Rectangle2D.Double(x, y - corr, nodeSize - corr,
					nodeSize));
		}
		if (c3 != null) {
			g2d.setColor(c3);
			g2d.fill(new Rectangle2D.Double(x - corr, y, nodeSize, nodeSize
					- corr));
		}
		if (n.isFixed())
			g2d.setColor(Color.yellow);
		else
			g2d.setColor(Color.black);
		g2d.draw(nodeR);

	}

	
	public static void paintPath(Graphics2D g2d, int x1, int y1, int x2,
			int y2, double orientation, double offset, boolean arrow) {
		paintPath(g2d, x1, y1, x2, y2, orientation, offset, arrow, false, 0.0, false);
	}

	public static void paintPath(Graphics2D g2d, int x1, int y1, int x2,
			int y2, double orientation, double offset, boolean arrow,
			boolean labelEdge, double confidence, boolean isHighlighted) {

		// First, the line should be drawn. When this is done,
		// calculate the angle of the arrow
		if (orientation != 0.0) {

			double[] cPoints = controlPoints(x1, y1, x2, y2, orientation,
					offset);

			CubicCurve2D.Double path = new CubicCurve2D.Double(x1, y1,
					cPoints[0], cPoints[1], cPoints[2], cPoints[3], x2, y2);

			g2d.draw(path);
			CubicCurve2D leftHalf = new CubicCurve2D.Double();
			path.subdivide(leftHalf, null);
			
			if (labelEdge) {
				if (isHighlighted) {
					Color origninalColor = g2d.getColor();
					Composite originalComposite = g2d.getComposite();
					g2d.setComposite(AlphaComposite.getInstance(AlphaComposite.SRC_OVER, 1.0f));
					g2d.setColor(Color.black);
					DecimalFormat df = new DecimalFormat("0.00");
					g2d.drawString(df.format(confidence), (int)leftHalf.getX2(), (int)leftHalf.getY2());
					g2d.setColor(origninalColor);
					g2d.setComposite(originalComposite);
				}
				DecimalFormat df = new DecimalFormat("0.00");
				g2d.drawString(df.format(confidence), (int)leftHalf.getX2(), (int)leftHalf.getY2());
			}
				
		} else {
			g2d.drawLine(x1, y1, x2, y2);
			if (labelEdge) {
				if (isHighlighted) {
					Color origninalColor = g2d.getColor();
					Composite originalComposite = g2d.getComposite();
					g2d.setComposite(AlphaComposite.getInstance(AlphaComposite.SRC_OVER, 1.0f));
					g2d.setColor(Color.black);
					DecimalFormat df = new DecimalFormat("0.00");
					g2d.drawString(df.format(confidence), x1 - ((x1 - x2) / 2), y1
							- ((y1 - y2) / 2) );
					g2d.setColor(origninalColor);
					g2d.setComposite(originalComposite);
				}
				DecimalFormat df = new DecimalFormat("0.00");
				g2d.drawString(df.format(confidence), x1 - ((x1 - x2) / 2), y1
						- ((y1 - y2) / 2) );
			}
		}
		
		if (arrow)
			drawArrow(g2d, x1, y1, x2, y2, orientation, offset);
	}

	

	// this returns the control points for e.g. drawing paths in postscript
	// as int
	public static double[] controlPoints(int x1, int y1, int x2, int y2,
			double orientation, double offset) {
		double deltaX = (x2 - x1);
		double deltaY = (y2 - y1);
		double len = Math.sqrt(deltaX * deltaX + deltaY * deltaY);
		double radius = len / 4.0;
		double theta, theta2;
		theta = Math.atan(deltaY / deltaX);
		if (deltaX < 0.0) {
			theta = theta += Math.PI;
		}
		// else theta2=theta;
		theta2 = theta + Math.PI;
		theta += orientation * offset;
		theta2 -= orientation * offset;
		double[] result = new double[4];
		result[2] = x2 + Math.cos(theta2) * radius;
		result[3] = y2 + Math.sin(theta2) * radius;
		result[0] = x1 + Math.cos(theta) * radius;
		result[1] = y1 + Math.sin(theta) * radius;

		return result;
	}

	public static int[] intControlPoints(int x1, int y1, int x2, int y2,
			double orientation, double offset) {
		double[] db = controlPoints(x1, y1, x2, y2, orientation, offset);
		int[] ib = new int[4];
		ib[0] = (int) db[0];
		ib[1] = (int) db[1];
		ib[2] = (int) db[2];
		ib[3] = (int) db[3];
		return ib;
	}

	public static Shape diagonalCross(int x, int y, float len) {
		len = len/2;
		x += 10; // coordinate correction
		y += 10;
		float thick = 2;
		GeneralPath p0 = new GeneralPath();
		float sqrt2 = (float) Math.sqrt(2.0);
		p0.moveTo(-thick + x, thick + y);
		p0.lineTo(-len - thick + x, -len + thick + y);
		p0.lineTo(-len + thick + x, -len - thick + y);
		p0.lineTo(0.0f + x, -thick * sqrt2 + y);
		p0.lineTo(len - thick + x, -len - thick + y);
		p0.lineTo(len + thick + x, -len + thick + y);
		p0.lineTo(thick * sqrt2 + x, 0.0f + y);
		p0.lineTo(len + thick + x, len - thick + y);
		p0.lineTo(len - thick + x, len + thick + y);
		p0.lineTo(0.0f + x, thick * sqrt2 + y);
		p0.lineTo(-len + thick + x, len + thick + y);
		p0.lineTo(-len - thick + x, len - thick + y);
		p0.lineTo(-thick * sqrt2 + x, 0.0f + y);
		p0.closePath();

		return p0;
	}
	
	
	public static Shape mito(int x, int y, float len) {
		//correction
		x = (int)(x-len/10);
		len = len/4;
		
		GeneralPath path = new GeneralPath();
		
		path.moveTo((float)x, (float)y);
		path.curveTo((float)x, (float)y, (float)(x+len*0.5), (float)(y-len*2.5), (float)(x+len), (float)y);
		path.curveTo((float)(x+len), (float)y, (float)(x+len*1.5), (float)(y-len*2.5), (float)(x+len*2), (float)y);
		path.lineTo((float)(x+len*2.5), (float)y);
		path.curveTo((float)(x+len*2.5), (float)y, (float)(x+len*2), (float)(y+len*2.5), (float)(x+len*1.5), (float)y);
		path.curveTo((float)(x+len*1.5), (float)y, (float)(x+len), (float)(y+len*2.5), (float)(x+len*0.5), (float)y);
		path.curveTo((float)(x+len*0.5), (float)y, (float)x, (float)(y+len*2.5), (float)(x-len*0.5), (float)y);
		path.curveTo((float)(x-len*0.5), (float)y, (float)(x-len), (float)(y+len*2.5), (float)(x-len*1.5), (float)y);
		//path.curveTo(x-len*1.5, y, x-len*2, y+len, x-len*2.5, y);
		path.lineTo((float)(x-len), (float)y);
		//path.curveTo(x-len*2, y, x-len*1.5, y-len, x-len, y);
		path.curveTo((float)(x-len), (float)y, (float)(x-len*0.5), (float)(y-len*2.5), (float)x, (float)y);
		
		Shape mitoBorder = new Ellipse2D.Double(x-len*2, y-len*1.5, len*5, len*3);
		path.append(mitoBorder, false);
		
		return path;
	}
	
	public static Shape golgi(int x, int y, float len) {
		//correction
		y = (int)(y-len/15);
		len = 5*len/4;
		
		float xLen = len * 0.9f;
		float yLen = len / 0.8f;
		
		GeneralPath path = new GeneralPath();
		
		path.moveTo((float)(x - xLen/2), (float)y);
		path.curveTo((float)(x-xLen/2), (float)y, (float)x, (float)(y-yLen/10), (float)(x+xLen/2), (float)y);
		path.curveTo((float)(x+xLen/2), (float)y, (float)(x+xLen/2+xLen/10), (float)(y+yLen/20), (float)(x+xLen/2), (float)(y+yLen/10));
		path.curveTo((float)(x+xLen/2), (float)(y+yLen/10), (float)x, (float)(y-yLen/10), (float)(x-xLen/2), (float)(y+yLen/10));
		path.curveTo((float)(x-xLen/2), (float)(y+yLen/10), (float)(x-xLen/2-xLen/10), (float)(y+yLen/20), (float)(x-xLen/2), (float)y);
		
		path.moveTo((float)(x-xLen/2-xLen/10), (float)(y-yLen/10));
		path.curveTo((float)(x-xLen/2-xLen/10), (float)(y-yLen/10), (float)x, (float)(y-yLen/10-yLen/10), (float)(x+xLen/2+xLen/10), (float)(y-yLen/10));
		path.curveTo((float)(x+xLen/2+xLen/10), (float)(y-yLen/10), (float)(x+xLen/2+xLen/10+xLen/10), (float)(y+xLen/20-xLen/10), (float)(x+xLen/2+xLen/10), (float)y);
		path.curveTo((float)(x+xLen/2+xLen/10), (float)y, (float)x, (float)(y-yLen/10-yLen/10), (float)(x-xLen/2-xLen/10), (float)y);
		path.curveTo((float)(x-xLen/2-xLen/10), (float)y, (float)(x-xLen/2-xLen/10-xLen/10), (float)(y+yLen/20-yLen/10), (float)(x-xLen/2-xLen/10), (float)(y-yLen/10));
		
		path.moveTo((float)(x-xLen/2+xLen/10), (float)(y+yLen/10));
		path.curveTo((float)(x-xLen/2+xLen/10), (float)(y+yLen/10), (float)x, (float)y, (float)(x+xLen/2-xLen/10), (float)(y+yLen/10));
		path.curveTo((float)(x+xLen/2-xLen/10), (float)(y+yLen/10), (float)(x+xLen/2), (float)(y+yLen/20+yLen/10), (float)(x+xLen/2-xLen/10), (float)(y+yLen/10+yLen/10));
		path.curveTo((float)(x+xLen/2-xLen/10), (float)(y+yLen/10+yLen/10), (float)x, (float)y, (float)(x-xLen/2+xLen/10), (float)(y+yLen/10+yLen/10));
		path.curveTo((float)(x-xLen/2+xLen/10), (float)(y+yLen/10+yLen/10), (float)(x-xLen/2), (float)(y+yLen/20+yLen/10), (float)(x-xLen/2+xLen/10), (float)(y+yLen/10));
		
		path.moveTo((float)(x-xLen/2+xLen/5), (float)y+yLen/5);
		path.curveTo((float)(x-xLen/2+xLen/5), (float)(y+yLen/5), (float)x, (float)(y+yLen/10), (float)(x+xLen/2-xLen/5), (float)(y+yLen/5));
		path.curveTo((float)(x+xLen/2-xLen/5), (float)(y+yLen/5), (float)(x+xLen/2-xLen/10), (float)(y+yLen/20+yLen/5), (float)(x+xLen/2-xLen/5), (float)(y+yLen/10+yLen/5));
		path.curveTo((float)(x+xLen/2-xLen/5), (float)(y+yLen/10+yLen/5), (float)x, (float)(y+yLen/10), (float)(x-xLen/2+xLen/5), (float)(y+yLen/10+yLen/5));
		path.curveTo((float)(x-xLen/2+xLen/5), (float)(y+yLen/10+yLen/5), (float)(x-xLen/2+xLen/10), (float)(y+yLen/20+yLen/5), (float)(x-xLen/2+xLen/5), (float)(y+yLen/5));
	
		return path;
	}
	
	public static Shape nucleus(int x, int y, float len) {
		Shape nucl = new Ellipse2D.Double(x-len/2,y-len/2, len, len);
		Shape nucleolus = new Ellipse2D.Double(x-len/2+len/5, y-len/2+len/5, len/10, len/10);

		GeneralPath path = new GeneralPath();
		
		//path.moveTo(x, y);
		path.append(nucl, false);
		path.append(nucleolus, false);
	
		return path;
	}
	
	public static Shape ER(int x, int y, float len) {
		//correction
		len = 5*len/4;
		
		GeneralPath path = new GeneralPath();
		
		path.moveTo((float)x, (float)y);
		
		path.lineTo((float)(x+len*0.5), (float)y);
		path.curveTo((float)(x+len*0.5), (float)y, (float)(x+len*0.5+len/10), (float)(y+len/40), (float)(x+len*0.5), (float)(y+len/20));
		path.lineTo((float)(x+len*0.2),(float)(y+len/20));
		path.lineTo((float)(x+len*0.2), (float)(y+len/10));
		path.lineTo((float)(x+len*0.4),(float)(y+len/10));
		path.curveTo((float)(x+len*0.4), (float)(y+len/10), (float)(x+len*0.4+1.5*len/20), (float)(y+len/10+len/40), (float)(x+len*0.4), (float)(y+3*len/20));
		path.lineTo((float)(x+len*0.1), (float)(y+3*len/20));
		path.lineTo((float)(x+len*0.1), (float)(y+len/5));
		path.lineTo((float)(x+len*0.45), (float)(y+len/5));
		path.curveTo((float)(x+len*0.45), (float)(y+len/5), (float)(x+len*0.45+1.5*len/20), (float)(y+len/5+len/40), (float)(x+len*0.45), (float)(y+len/4));
		path.lineTo((float)(x-len*0.5), (float)(y+len/4));
		path.curveTo((float)(x-len*0.5), (float)(y+len/4), (float)(x-len*0.5-1.5*len/20), (float)(y+len/5+len/40), (float)(x-len*0.5), (float)(y+len/5));
		path.lineTo((float)(x+len/20), (float)(y+len/5));
		path.lineTo((float)(x+len/20), (float)(y+3*len/20));
		path.lineTo((float)(x-len*0.4), (float)(y+3*len/20));
		path.curveTo((float)(x-len*0.4), (float)(y+3*len/20), (float)(x-len*0.4-1.5*len/20), (float)(y+len/10+len/40), (float)(x-len*0.4), (float)(y+len/10));
		path.lineTo((float)(x+len*0.15), (float)(y+len/10));
		path.lineTo((float)(x+len*0.15), (float)(y+len/20));
		path.lineTo((float)(x-len*0.2), (float)(y+len/20));
		path.curveTo((float)(x-len*0.2), (float)(y+len/20), (float)(x-len*0.2-1.5*len/20), (float)(y+len/40), (float)(x-len*0.2), (float)y);
		path.lineTo((float)(x-len*0.05), (float)y);
		path.lineTo((float)(x-len*0.05), (float)(y-len/20));
		path.lineTo((float)(x-len*0.3), (float)(y-len/20));
		path.curveTo((float)(x-len*0.3), (float)(y-len/20), (float)(x-len*0.3-1.5*len/20), (float)(y-len/20-len/40), (float)(x-len*0.3), (float)y-len/10);
		path.lineTo((float)(x+len*0.2), (float)(y-len/10));
		path.lineTo((float)(x+len*0.2), (float)(y-3*len/20));
		path.lineTo((float)(x-len*0.45), (float)(y-3*len/20));
		path.curveTo((float)(x-len*0.45), (float)(y-3*len/20), (float)(x-len*0.45-1.5*len/20), (float)(y-3*len/20-len/40), (float)(x-len*0.45), (float)y-len/5);
		path.lineTo((float)x, (float)(y-len/5));
		path.lineTo((float)x, (float)(y-len/4));
		path.lineTo((float)(x-len*0.2), (float)(y-len/4));
		path.curveTo((float)(x-len*0.2), (float)(y-len/4), (float)(x-len*0.2-1.5*len/20), (float)(y-len/4-len/40), (float)(x-len*0.2), (float)y-len/4-len/20);
		path.lineTo((float)(x+len*0.2), (float)(y-len/4-len/20));
		path.curveTo((float)(x+len*0.2), (float)(y-len/4-len/20), (float)(x+len*0.2+1.5*len/20), (float)(y-len/4-len/40), (float)(x+len*0.2), (float)y-len/4);
		path.lineTo((float)(x+len*0.05), (float)(y-len/4));
		path.lineTo((float)(x+len*0.05), (float)(y-len/5));
		path.lineTo((float)(x+len*0.45), (float)(y-len/5));
		path.curveTo((float)(x+len*0.45), (float)(y-len/5), (float)(x+len*0.45+1.5*len/20), (float)(y-len/5+len/40), (float)(x+len*0.45), (float)y-3*len/20);
		path.lineTo((float)(x+len*0.25), (float)(y-3*len/20));
		path.lineTo((float)(x+len*0.25), (float)(y-len/10));
		path.lineTo((float)(x+len*0.35), (float)(y-len/10));
		path.curveTo((float)(x+len*0.35), (float)(y-len/10), (float)(x+len*0.35+1.5*len/20), (float)(y-len/20-len/40), (float)(x+len*0.35), (float)y-len/20);
		path.lineTo((float)x, (float)(y-len/20));
		path.lineTo((float)x, (float)y);
		
		return path;
	}
	

	public static Polygon revTriangle(int x, int y, int nodeSize) {
		int x1 = (int) (x + nodeSize / 2);
		int x2 = (int) (x + nodeSize);
		int y2 = (int) (y + nodeSize);
		int[] xpoints = { x, x1, x2 };
		int[] ypoints = { y, y2, y };
		int npoints = 3;
		return new Polygon(xpoints, ypoints, npoints);
	}
	
	 /**
     * Returns a left-pointing triangle of the given dimenisions.
     */
    public static Shape leftTriangle(float x, float y, float height) {
    	GeneralPath path = new GeneralPath();
    	path.reset();
        path.moveTo(x+height, y);
        path.lineTo(x+height, y+height);
        path.lineTo(x, y+height/2);
        path.closePath();
        return path;
    }
    
    /**
     * Returns a right-pointing triangle of the given dimenisions.
     */
    public static Shape rightTriangle(float x, float y, float height) {
    	GeneralPath path = new GeneralPath();
    	path.reset();
        path.moveTo(x,y+height);
        path.lineTo(x+height, y+height/2);
        path.lineTo(x, y);
        path.closePath();
        return path;
    }

    public static Shape cross(float x, float y, float height) {
        float h14 = 3*height/8, h34 = 5*height/8;
        GeneralPath path = new GeneralPath();
        path.reset();
        path.moveTo(x+h14, y);
        path.lineTo(x+h34, y);
        path.lineTo(x+h34, y+h14);
        path.lineTo(x+height, y+h14);
        path.lineTo(x+height, y+h34);
        path.lineTo(x+h34, y+h34);
        path.lineTo(x+h34, y+height);
        path.lineTo(x+h14, y+height);
        path.lineTo(x+h14, y+h34);
        path.lineTo(x, y+h34);
        path.lineTo(x, y+h14);
        path.lineTo(x+h14, y+h14);
        path.closePath();
        return path;
    }

    /**
     * Returns a star shape of the given dimenisions.
     */
    public static Shape star(float x, float y, float height) {
        float s = (float)(height/(2*Math.sin(Math.toRadians(54))));
        float shortSide = (float)(height/(2*Math.tan(Math.toRadians(54))));
        float mediumSide = (float)(s*Math.sin(Math.toRadians(18)));
        float longSide = (float)(s*Math.cos(Math.toRadians(18)));
        float innerLongSide = (float)(s/(2*Math.cos(Math.toRadians(36))));
        float innerShortSide = innerLongSide*(float)Math.sin(Math.toRadians(36));
        float innerMediumSide = innerLongSide*(float)Math.cos(Math.toRadians(36));

        GeneralPath path = new GeneralPath();
        path.reset();
        path.moveTo(x, y+shortSide);            
        path.lineTo((x+innerLongSide),(y+shortSide));
        path.lineTo((x+height/2),y);
        path.lineTo((x+height-innerLongSide),(y+shortSide));
        path.lineTo((x+height),(y+shortSide));
        path.lineTo((x+height-innerMediumSide),(y+shortSide+innerShortSide));        
        path.lineTo((x+height-mediumSide),(y+height));
        path.lineTo((x+height/2),(y+shortSide+longSide-innerShortSide));
        path.lineTo((x+mediumSide),(y+height));
        path.lineTo((x+innerMediumSide),(y+shortSide+innerShortSide));
        path.closePath();
        return path;
    }

    /**
     * Returns a hexagon shape of the given dimenisions.
     */
    public static Shape hexagon(float x, float y, float height) {
        float width = height/2;  
        GeneralPath path = new GeneralPath();
        path.reset();
        path.moveTo(x,            y+0.5f*height);
        path.lineTo(x+0.5f*width, y);
        path.lineTo(x+1.5f*width, y);
        path.lineTo(x+2.0f*width, y+0.5f*height);
        path.lineTo(x+1.5f*width, y+height);
        path.lineTo(x+0.5f*width, y+height);
        path.closePath();      
        return path;
    }

}
