// Medusa is a graph viewer which allows interactive editing of networks
// (edges and nodes) and also connects to databases.
//
// Copyright (C) 2006 Sean Hooper
//
// This program is free software; you can redistribute it and/or modify 
// it under the terms of the GNU General Public License as published by 
// the Free Software Foundation; either version 2 of the License, or (at 
// your option) any later version.
//
// This program is distributed in the hope that it will be useful, but 
// WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General 
// Public License for more details.
//
// You should have received a copy of the GNU General Public License along 
// with this program; if not, write to the 
// Free Software Foundation, Inc., 
// 59 Temple Place, Suite 330, 
// Boston, MA 02111-1307 USA

package medusa;

import java.awt.Color;
import java.util.Hashtable;
import java.util.Iterator;
import java.io.*;
import java.util.regex.*;

public class MedusaSettings {

	// Table of colors, which allows colors to be changed
	Hashtable colorTable = null;

	// And also the interaction names.
	Hashtable interactionTable = null;
	
	Hashtable checkedTable = null;
	
	int indexOffset = 0;

	Color[] colorList = { new Color(0, 180, 0), new Color(255, 0, 0),
			new Color(0, 0, 255), new Color(140, 140, 255),
			new Color(30, 30, 30), new Color(200, 0, 200),
			new Color(0, 160, 200), new Color(170, 200, 0), randomColor(),
			randomColor(), randomColor(), randomColor() };

	// public Iterator''
	

	public Color randomColor() {
		int r = (int) (Math.random() * 255);
		int g = (int) (Math.random() * 255);
		int b = Math.min(255, 255 * 2 - r - g);
		// System.out.println(r+" "+g+" "+b);
		return new Color(r, g, b);
	}

	public MedusaSettings(Hashtable colorTable, Hashtable interactionTable, Hashtable checkedTable) {
		//init();
		this.colorTable = colorTable;
		this.interactionTable = interactionTable;
		this.checkedTable = checkedTable;
	}
	
	public MedusaSettings() {
		interactionTable = new Hashtable();
		colorTable = new Hashtable();
		checkedTable = new Hashtable();
	}

	public Color getColorFromParam(String param) {
		String[] rgb = param.split(",");
		int red, green, blue;
		red = Integer.parseInt(rgb[0]);
		green = Integer.parseInt(rgb[1]);
		blue = Integer.parseInt(rgb[2]);
		return new Color(red, green, blue);
	}

	public MedusaSettings(String settingsParam) {
		// handles a line of colors for the applet version
		// this requires correct input!
		if (settingsParam != null) {
			// System.out.println("debug Stringsettings "+settingsParam);
			//readParam(settingsParam);
		} 
		else {
			init();
		}
	}
	
	public MedusaSettings(String settingsParam, int offset) {
		this.indexOffset = offset;
		if (settingsParam != null) {
			//readParam(settingsParam);
		} 
			else {
			init();
		}
		
	}

//	public void readParam(String settingsParam) {
//		String[] interactions = settingsParam.split(";");
//		Color c;
//		for (int i = 0; i < interactions.length; ++i) {
//			String[] interaction = interactions[i].split(":");
//			if (interaction.length < 2) {
//				c = getColorFromParam(interaction[0]);
//			} else {
//				c = getColorFromParam(interaction[1]);
//				interactionTable.put(new Integer(i + 1 + indexOffset), interaction[0]);
//			}
//			colorTable.put(new Integer(i + 1 + indexOffset), c);
//		}
//
//		/*
//		 * for (int i=0; i<colors.length; i++){ c=getColorFromParam(colors[i]);
//		 * //System.out.println(c); colorTable.put(new Integer(i+1),c); }
//		 */
//	}
	
	public Hashtable getCheckedTable() {
		return this.checkedTable;
	}
	
	public Hashtable getInteractionTable() {
		return this.interactionTable;
	}
	
	public void setInteractionTable(Hashtable table) {
		this.interactionTable = table;
	}
	
	public void setColorTable(Hashtable table) {
		this.colorTable = table;
	}

	public void clear() {
		colorTable.clear();
		interactionTable.clear();

	}

	public void init() {
		// clear();
		initColorTable();
		initInteractionTable();
	}

	public void initCOGS() {
		// clear();
		initColorTable();
		initCOGInteractionTable();
	}

	public void initColorTable() {
		for (int i = 0; i < colorList.length; i++)
			colorTable.put(new Integer(i + 1), colorList[i]);
	}

	public void initInteractionTable() {
		interactionTable.put(new Integer(1), "Neighbourhood");
		interactionTable.put(new Integer(2), "Gene Fusion");
		interactionTable.put(new Integer(3), "Cooccurence");
		interactionTable.put(new Integer(4), "Homology");
		interactionTable.put(new Integer(5), "Coexpression");
		interactionTable.put(new Integer(6), "Experiments");
		interactionTable.put(new Integer(7), "Databases");
		interactionTable.put(new Integer(8), "Text Mining");
	}

	public void initCOGInteractionTable() {
		interactionTable.put(new Integer(1), "Neighbourhood");
		interactionTable.put(new Integer(2), "Gene Fusion");
		// interactionTable.put(new Integer(3),"Cooccurence");
		interactionTable.put(new Integer(3), "Homology");
		interactionTable.put(new Integer(4), "Coexpression");
		interactionTable.put(new Integer(5), "Experiments");
		interactionTable.put(new Integer(6), "Databases");
		interactionTable.put(new Integer(7), "Text Mining");
	}

	public void load(String fileName) throws IOException {
		File loadFile = new File(fileName);
		BufferedReader in = new BufferedReader(new InputStreamReader(
				new FileInputStream(loadFile)));
		String inLine;
		// clear settings
		Pattern xyPattern = Pattern.compile("(\\d+)[, ](\\d+)[, ](\\d+)");

		interactionTable = new Hashtable();
		colorTable = new Hashtable();
		initColorTable();
		// load new settings
		while ((inLine = in.readLine()) != null) {
			String result[] = inLine.split("\\t");
			Matcher matcher = xyPattern.matcher(inLine);
			// set names
			interactionTable.put(new Integer(result[0]), result[1]);
			// set color, if present
			if (matcher.find()) {
				int red = Integer.parseInt(matcher.group(1));
				int green = Integer.parseInt(matcher.group(2));
				int blue = Integer.parseInt(matcher.group(3));
				colorTable.put(new Integer(result[0]), new Color(red, green,
						blue));
			}

		}
		in.close();
	}

	public int getSize() {
		return interactionTable.size();
	}

	public int getSizeColorTable() {
		return colorTable.size();
	}

	public void report() {
		for (int i = 1; i < interactionTable.size(); i++) {
			System.out.println(getName(new Integer(i), 0));
		}
	}

	public Color parseColor(String rgb) {
		String result[] = rgb.split(",");
		if (result.length < 3)
			return randomColor();
		int r = Integer.parseInt(result[0]);
		int g = Integer.parseInt(result[1]);
		int b = Integer.parseInt(result[2]);
		return new Color(r, g, b);
	}

	public Color getColor(Integer i) {
		return (Color) colorTable.get(i);
	}

	public void setColor(Integer i, Color color) {
		colorTable.put(i, color);
	}

	public String getName(Integer i, int len) {
		String name = (String) interactionTable.get(i);
		if (name == null)
			return "Group: " + i;
		// System.out.println(name.length());
		if ((name.length() < len) || (len == 0))
			return name;
		return name.substring(0, len - 1);
	}

	public String getName(Integer i) {
		String name = (String) interactionTable.get(i);
		if (name == null)
			return "Group: " + i;
		// System.out.println(name.length());

		return name;

	}

	public void setName(Integer i, String name) {
		interactionTable.put(i, name);
	}
	
	public boolean containsName(String name){
		return interactionTable.containsValue(name);
	}
	
	public int getIndexOfName(String name){
		for (Iterator iter = interactionTable.keySet().iterator(); iter.hasNext();) {
			Integer key = (Integer) iter.next();
			String element = (String)interactionTable.get(key);
			
			if(element.equalsIgnoreCase(name))
				return key.intValue();
		}
		return -1;
	}

}
